#include "types.h"

#include <maps/libs/common/include/exception.h>

#include <boost/bimap.hpp>
#include <ostream>

namespace maps {

namespace geolib3 {

std::ostream& operator<<(std::ostream& out, const BoundingBox& box)
{
    return out << "[[" << box.minX() << "," << box.minY() << "]"
               << ",[" << box.maxX() << "," << box.maxY() << "]]";
}

} // geolib3

namespace mrc {
namespace toloka {

namespace {

// These string values come from the frontend
std::map<std::string, TaskAnswer> TASK_ANSWER_MAP {
    {"not", TaskAnswer::Ok},
    {"not-classified", TaskAnswer::NotClassified},
    {"not-recognized", TaskAnswer::NotRecognized},
    {"not-signs", TaskAnswer::NotSign},
    {"not-photo", TaskAnswer::NotLoaded},
};


} // anonymous namespace

bool operator==(const TaskOutput& lhs, const TaskOutput& rhs)
{
    return std::tie(lhs.answer, lhs.signId) ==
            std::tie(rhs.answer, rhs.signId);
}

bool operator<(const TaskOutput& lhs, const TaskOutput& rhs)
{
    return std::tie(lhs.answer, lhs.signId) <
            std::tie(rhs.answer, rhs.signId);
}

TaskAnswer parseTaskAnswer(const std::string& answer)
{
    auto itr = TASK_ANSWER_MAP.find(answer);
    REQUIRE(itr != TASK_ANSWER_MAP.end(),
            "Invalid task answer: " << answer);
    return itr->second;
}

std::string toString(TaskAnswer answer)
{
    switch (answer) {
        case TaskAnswer::Ok:
            return "ok";
        case TaskAnswer::NotClassified:
            return "not-classified";
        case TaskAnswer::NotRecognized:
            return "not-recognized";
        case TaskAnswer::NotSign:
            return "not-a-sign";
        case TaskAnswer::NotLoaded:
            return "not-loaded";
        default:
            throw maps::RuntimeError() << "Invalid task answer: "
                << static_cast<std::underlying_type<TaskAnswer>::type>(answer);
    }
}

std::ostream& operator<<(std::ostream& out, TaskAnswer answer)
{
    return out << toString(answer);
}

std::ostream& operator<<(std::ostream& out, const TaskInput& taskInput)
{
    return out << "{source: " << taskInput.source
               << ", bbox: " << taskInput.bbox
               << "}";
}

std::ostream& operator<<(std::ostream& out, const TaskOutput& taskOutput)
{
    return out << "{rectangles: " << taskOutput.answer
               << ", signId: " << taskOutput.signId
               << "}";
}

std::ostream& operator<<(std::ostream& out, const TaskResult& taskResult)
{
    return out << "{ input: " << taskResult.input
               << ", output: " << taskResult.output
               << " }";
}

std::ostream& operator<<(std::ostream& out, const TaskSuite& taskSuite)
{
    return out << "{ id: " << taskSuite.id
               << ", poolId: " << taskSuite.poolId
               << ", overlap: " << taskSuite.overlap
               << ", tasks: " << taskSuite.tasks
               << " }";
}

std::ostream& operator<<(std::ostream& out, const AssignmentResult& result)
{
    return out << "{ inputs: " << result.inputs
               << ", outputs: " << result.outputs
               << ", status: " << result.status
               << ", taskSuiteId: " << result.taskSuiteId
               << ", assignmentId: " << result.assignmentId
               << ", userId: " << result.userId
               << " }";

}
std::ostream& operator<<(std::ostream& out,
                         const TaskSuiteIdToResults& idToResults)
{
    out << "{ ";
    wiki::common::join(idToResults,
            [](const TaskSuiteIdToResults::value_type& pair) {
                std::ostringstream oss;
                oss << pair.first << " -> {" << pair.second << "}";
                return oss.str();
            }, ",");
    out << " }";
    return out;
}

std::ostream& operator<<(std::ostream& out, const UserStat& userStat)
{
    return out << "{ userId: " << userStat.userId
               << ", assignmentId: " << userStat.assignmentId
               << ", assignmentStatus: " << userStat.assignmentStatus
               << ", tasksCount: " << userStat.tasksCount
               << ", correctCount: " << userStat.correctCount
               << " }";
}

} // toloka
} // mrc
} // maps
