#pragma once

#include <maps/libs/geolib/include/bounding_box.h>
#include <yandex/maps/mrc/toloka_client/assignment.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <iosfwd>
#include <string>
#include <unordered_map>
#include <vector>

namespace maps {

namespace geolib3 {

std::ostream& operator<<(std::ostream& out, const BoundingBox& box);

} // geolib3

using Bbox = geolib3::BoundingBox;

namespace mrc {
namespace toloka {

enum class TaskAnswer {
    Ok,
    NotRecognized,
    NotClassified,
    NotSign,
    NotLoaded
};

// Input parameters of a single task of sign classification
struct TaskInput
{
    std::string source;
    Bbox bbox;

    TaskInput(std::string source_, Bbox bbox_)
        : source(std::move(source_))
        , bbox(std::move(bbox_))
    {}
};

// Output parameters of a single task of sign classification
struct TaskOutput
{
    TaskAnswer answer;
    std::string signId;
    TaskOutput(TaskAnswer answer_, std::string signId_)
        : answer(std::move(answer_))
        , signId(std::move(signId_))
    {}
};

/// Returns true if 'answer's and 'signId's in both objects are equal,
///         false otherwise
bool operator==(const TaskOutput& lhs, const TaskOutput& rhs);

/// Uses ordering first by 'answer', then by 'signId'
bool operator<(const TaskOutput& lhs, const TaskOutput& rhs);

// Result of a sign classification task
struct TaskResult
{
    TaskInput input;
    TaskOutput output;
};

using TaskResults = std::vector<TaskResult>;


struct TaskSuite
{
    std::string id;
    std::string poolId;
    size_t overlap;
    std::vector<TaskInput> tasks;
};

using TaskSuites = std::vector<TaskSuite>;
using IdToTaskSuite = std::unordered_map<std::string, TaskSuite>;

// Raw result of a task suite from a single user
struct AssignmentResult
{
    std::string taskSuiteId;
    std::string assignmentId;
    io::AssignmentStatus status;
    std::string userId;
    std::vector<TaskInput> inputs;
    std::vector<TaskOutput> outputs;
};

using AssignmentResults = std::vector<AssignmentResult>;

// Map from task suite ID to collection of results from multiple users
using TaskSuiteIdToResults
        = std::unordered_map<std::string, AssignmentResults>;


/**
 * User's statistic for a single task suite
 */
struct UserStat
{
    UserStat(const AssignmentResult& result)
        : userId(result.userId)
        , assignmentId(result.assignmentId)
        , assignmentStatus(result.status)
        , tasksCount(result.inputs.size())
        , correctCount(0)
    {}

    std::string userId;
    std::string assignmentId;
    io::AssignmentStatus assignmentStatus;
    size_t tasksCount; // Total number of tasks in the task suite
    size_t correctCount; // Number of tasks correctly answered by the user
};

using UserIdToStat = std::unordered_map<std::string, UserStat>;

// Result of processing task suite:
// @param taskResults - merged results for each task in the suite
// @param userIdToStat - per-user statistic
struct TaskSuiteResult
{
    TaskResults taskResults;
    UserIdToStat userIdToStat;
};


TaskAnswer parseTaskAnswer(const std::string& answer);
std::string toString(TaskAnswer answer);

template <typename StreamType, typename T>
StreamType& operator<<(StreamType& out, const std::vector<T>& vec)
{
    return out << "[ " << wiki::common::join(vec, ", ") << " ]";
}

std::ostream& operator<<(std::ostream& out, TaskAnswer answer);
std::ostream& operator<<(std::ostream& out, const TaskInput& taskInput);
std::ostream& operator<<(std::ostream& out, const TaskOutput& taskOutput);
std::ostream& operator<<(std::ostream& out, const TaskResult& taskResult);
std::ostream& operator<<(std::ostream& out, const TaskSuite& taskSuite);
std::ostream& operator<<(std::ostream& out, const AssignmentResult& result);
std::ostream& operator<<(std::ostream& out, const TaskSuiteIdToResults& idToResults);
std::ostream& operator<<(std::ostream& out, const UserStat& userStat);

} // toloka
} // mrc
} // maps
