#include "request.h"

#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/mrc/toloka_client/assignment.h>
#include <yandex/maps/mrc/toloka_client/filter.h>
#include <yandex/maps/mrc/toloka_client/task_suite.h>

namespace maps {
namespace mrc {
namespace toloka {

namespace {

const size_t BATCH_SIZE = 20;

const std::string FIELD_POLYGONS = "polygons";
const std::string FIELD_SOURCE = "source";

Rectangles rectanglesFromJson(const json::Value& jsonRectangles)
{
    REQUIRE(jsonRectangles.isArray(), "Bad format of 'polygons' parameter");

    Rectangles rectangles;
    for (const auto& rectangle : jsonRectangles) {
        REQUIRE(rectangle.isArray() && rectangle.size() == 2,
                "Bad polygon coordinates count");
        auto p1 = rectangle[0];
        auto p2 = rectangle[1];
        REQUIRE(p1.isArray() && p1.size() == 2, "Bad coordinate format");
        REQUIRE(p2.isArray() && p2.size() == 2, "Bad coordinate format");
        auto x1 = p1[0].as<double>();
        auto y1 = p1[1].as<double>();
        auto x2 = p2[0].as<double>();
        auto y2 = p2[1].as<double>();
        rectangles.emplace_back(geolib3::Point2(x1, y1),
                                geolib3::Point2(x2, y2));
    }
    return rectangles;
}


std::vector<TaskInput> getTasksInputs(const io::TaskSuiteItems& items)
{
    std::vector<TaskInput> taskInputs;
    for (const auto& item : items) {
        TaskInput input;
        input.source = item.inputValues()[FIELD_SOURCE].as<std::string>();
        taskInputs.push_back(std::move(input));
    }
    return taskInputs;
}

std::vector<TaskOutput> getTasksOutputs(const io::AssignmentSolutions& items)
{
    std::vector<TaskOutput> taskOutputs;
    for (const auto& item : items) {
        const auto& ov = item.outputValues();
        if (ov.hasField(FIELD_POLYGONS) && !ov[FIELD_POLYGONS].isNull()) {
            auto polygonsStr = ov[FIELD_POLYGONS].as<std::string>();
            taskOutputs.emplace_back(
                rectanglesFromJson(json::Value::fromString(polygonsStr)));
        }
    }
    return taskOutputs;
}


} // anonymous namespace

IdToTaskSuite loadTaskSuites(const io::TolokaClient& tolokaClient,
                             const std::string& poolId)
{
    INFO() << "Loading task suites";
    IdToTaskSuite resultMap;

    io::Filter filter;
    filter.byPoolId(poolId);
    auto resp = tolokaClient.getTaskSuites(std::move(filter));

    for (const auto& ts : resp.taskSuites()) {
        TaskSuite taskSuite;
        taskSuite.id = ts.id();
        taskSuite.poolId = ts.poolId();
        taskSuite.overlap = ts.overlap();
        taskSuite.tasks = getTasksInputs(ts.tasks());
        resultMap.emplace(ts.id(), std::move(taskSuite));
    }

    return resultMap;
}

TaskSuiteIdToResults
loadTaskSuitesResults(const io::TolokaClient& tolokaClient,
                      const std::string& poolId)
{
    INFO() << "Loading task suite results";
    TaskSuiteIdToResults resultMap;
    io::Filter filter;
    filter.byPoolId(poolId);
    auto resp = tolokaClient.getAssignments(std::move(filter));

    for (const auto& assignment : resp.assignments()) {
        if (assignment.status() != io::AssignmentStatus::Accepted &&
            assignment.status() != io::AssignmentStatus::Submitted &&
            assignment.status() != io::AssignmentStatus::Rejected) {
            continue;
        }

        AssignmentResult result;
        result.taskSuiteId = assignment.taskSuiteId();
        result.assignmentId = assignment.id();
        result.status = assignment.status();

        result.userId = assignment.userId();
        result.inputs = getTasksInputs(assignment.tasks());
        result.outputs = getTasksOutputs(assignment.solutions());

        resultMap[assignment.taskSuiteId()].push_back(std::move(result));
    }
    return resultMap;
}

} // toloka
} // mrc
} // maps

