#include "tool.h"

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <utility>
#include <fstream>

namespace {

maps::mrc::db::TIds readFeatureIds(const std::string& path)
{
    maps::mrc::db::TIds result;
    std::ifstream file(path);
    maps::mrc::db::TId id;
    while (file >> id) {
        result.push_back(id);
    }
    return result;
}

} // anonymous namespace

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser;
    auto config = parser.string("config")
        .help("path to configuration");
    auto poolId = parser.string("toloka-pool-id")
        .help("toloka pool ID to post task suites to")
        .required();
    auto inputFile = parser.string("input-file")
        .required()
        .help("path to file with feature IDs sorted in ascending order");
    auto startFeatureId = parser.num("start-feature-id")
        .help("feature ID to start with");
    auto taskSuitesCount = parser.num("task-suites-count")
        .help("number of task suites to create, default is 1")
        .defaultValue(1);
    auto overlap = parser.num("overlap")
        .help("number of Toloka users handling each tasks, default is 5")
        .defaultValue(5);
    parser.parse(argc, argv);

    std::string configPath = maps::mrc::common::CONFIG_PATH;
    if (config.defined()) {
        configPath = config;
    }
    maps::mrc::common::Config cfg{configPath};
    auto pool = cfg.makePoolHolder();
    auto txn = pool.pool().slaveTransaction();
    size_t count = taskSuitesCount;

    maps::mrc::toloka::io::TolokaClient tolokaClient(
        cfg.tolokaConfig().host(),
        cfg.tolokaConfig().authHeader());
    tolokaClient.setTimeout(std::chrono::seconds(60))
                .setMaxRequestAttempts(3)
                .setRetryInitialTimeout(std::chrono::seconds(1))
                .setRetryTimeoutBackoff(2);

    auto ids = readFeatureIds(inputFile);
    REQUIRE(std::is_sorted(ids.begin(), ids.end()),
            "Ids in input file are not sorted");

    maps::mrc::toloka::DetectionTasksGenerator tasksGen(*txn, tolokaClient, poolId);

    auto beg = std::lower_bound(ids.begin(), ids.end(), startFeatureId);
    auto next = tasksGen.generate(beg, ids.end(), count, overlap);

    if (next != ids.end()) {
        INFO() << "Stopped at feature id: " << *next;
    } else {
        INFO() << "All features processed";
    }

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    ERROR() << "unknown error";
    return EXIT_FAILURE;
}
