import os
import copy
import json
import difflib
from bs4 import BeautifulSoup


def load_text(project_dir, name):
    path = os.path.join(project_dir, name)
    assert(os.path.isfile(path)), 'There is no {}'.format(path)
    return open(path).read()


def save_text(text, project_dir, name):
    path = os.path.join(project_dir, name)
    with open(path, 'w') as f:
        f.write(text)


def load_json(project_dir, name):
    path = os.path.join(project_dir, name)
    assert(os.path.isfile(path)), 'There is no {}'.format(path)
    return json.load(open(path))


def save_json(data, project_dir, name):
    path = os.path.join(project_dir, name)
    with open(path, 'w') as f:
        json.dump(data, f, indent=2, ensure_ascii=False, sort_keys=True)


def text_diff(a, b):
    a_lines = a.split('\n')
    b_lines = b.split('\n')
    return [l for l in difflib.ndiff(a_lines, b_lines) if l.strip() and l[0] in ['-', '+']]


def json_diff(a, b):
    a_text = json.dumps(a, indent=2, ensure_ascii=False, sort_keys=True)
    b_text = json.dumps(b, indent=2, ensure_ascii=False, sort_keys=True)
    return text_diff(a_text, b_text)


def add_json_diff(results, name, a, b):
    diff = json_diff(a, b)
    if diff:
        results[name] = diff


def add_text_diff(results, name, a, b):
    diff = text_diff(a, b)
    if diff:
        results[name] = diff


ID = 'id'
PROJECT = 'project'
PROJECT_FILENAME = 'project.json'

INSTRUCTIONS = 'public_instructions'
INSTRUCTIONS_FILENAME = 'instructions.html'

TASK_SPEC = 'task_spec'
INPUT_SPEC = 'input_spec'
INPUT_SPEC_FILENAME = 'input_spec.json'
OUTPUT_SPEC = 'output_spec'
OUTPUT_SPEC_FILENAME = 'output_spec.json'

VIEW_SPEC = 'view_spec'
MARKUP = 'markup'
MARKUP_FILENAME = 'markup.html'
SCRIPT = 'script'
SCRIPT_FILENAME = 'script.js'
STYLES = 'styles'
STYLES_FILENAME = 'styles.css'


class ProjectSpec:
    def from_json(self, spec_json):
        self.__spec = spec_json

        self.__template = copy.deepcopy(self.__spec)
        self.__template[ID] = "{{ PROJECT_ID }}"
        self.__template[INSTRUCTIONS] = "{{ INSTRUCTIONS }}"
        self.__template[TASK_SPEC][INPUT_SPEC] = "{{ INPUT_SPEC }}"
        self.__template[TASK_SPEC][OUTPUT_SPEC] = "{{ OUTPUT_SPEC }}"
        self.__template[TASK_SPEC][VIEW_SPEC][MARKUP] = "{{ MARKUP }}"
        self.__template[TASK_SPEC][VIEW_SPEC][SCRIPT] = "{{ SCRIPT }}"
        self.__template[TASK_SPEC][VIEW_SPEC][STYLES] = "{{ STYLES }}"

        return self

    def from_arcadia(self, project_id, project_dir):
        assert os.path.isdir(project_dir), '{} is not directory'.format(project_dir)
        self.__template = load_json(project_dir, PROJECT_FILENAME)

        self.__spec = copy.deepcopy(self.__template)
        self.__spec[ID] = str(project_id)
        self.__spec[INSTRUCTIONS] = load_text(project_dir, INSTRUCTIONS_FILENAME)
        self.__spec[TASK_SPEC][INPUT_SPEC] = load_json(project_dir, INPUT_SPEC_FILENAME)
        self.__spec[TASK_SPEC][OUTPUT_SPEC] = load_json(project_dir, OUTPUT_SPEC_FILENAME)
        self.__spec[TASK_SPEC][VIEW_SPEC][MARKUP] = load_text(project_dir, MARKUP_FILENAME)
        self.__spec[TASK_SPEC][VIEW_SPEC][SCRIPT] = load_text(project_dir, SCRIPT_FILENAME)
        self.__spec[TASK_SPEC][VIEW_SPEC][STYLES] = load_text(project_dir, STYLES_FILENAME)

        return self

    def template(self):
        return copy.deepcopy(self.__template.copy())

    def instructions(self):
        return BeautifulSoup(self.__spec[INSTRUCTIONS], features="lxml").prettify()

    def input_spec(self):
        return copy.deepcopy(self.__spec[TASK_SPEC][INPUT_SPEC])

    def output_spec(self):
        return copy.deepcopy(self.__spec[TASK_SPEC][OUTPUT_SPEC])

    def markup(self):
        return BeautifulSoup(self.__spec[TASK_SPEC][VIEW_SPEC][MARKUP], features="lxml").prettify()

    def script(self):
        return self.__spec[TASK_SPEC][VIEW_SPEC][SCRIPT]

    def styles(self):
        return self.__spec[TASK_SPEC][VIEW_SPEC][STYLES]

    def save(self, project_dir):
        if not os.path.isdir(project_dir):
            os.mkdir(project_dir)

        save_json(self.template(), project_dir, PROJECT_FILENAME)
        save_text(self.instructions(), project_dir, INSTRUCTIONS_FILENAME)
        save_json(self.input_spec(), project_dir, INPUT_SPEC_FILENAME)
        save_json(self.output_spec(), project_dir, OUTPUT_SPEC_FILENAME)
        save_text(self.markup(), project_dir, MARKUP_FILENAME)
        save_text(self.script(), project_dir, SCRIPT_FILENAME)
        save_text(self.styles(), project_dir, STYLES_FILENAME)

    def json(self):
        return self.__spec.copy()

    def str(self):
        return json.dumps(self.__spec, ensure_ascii=False, sort_keys=True)

    def diff(self, spec):
        results = {}
        add_json_diff(results, PROJECT, self.template(), spec.template())
        add_text_diff(results, INSTRUCTIONS, self.instructions(), spec.instructions())
        add_json_diff(results, INPUT_SPEC, self.input_spec(), spec.input_spec())
        add_json_diff(results, OUTPUT_SPEC, self.output_spec(), spec.output_spec())
        add_text_diff(results, MARKUP, self.markup(), spec.markup())
        add_text_diff(results, SCRIPT, self.script(), spec.script())
        add_text_diff(results, STYLES, self.styles(), spec.styles())
        return results
