#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_type_info.h>

#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/builder.h>

namespace {

std::map<std::string, maps::mrc::db::toloka::TaskType> TASK_TYPE_NAME_TO_TYPE {
    {
        "traffic-light-detection",
        maps::mrc::db::toloka::TaskType::TrafficLightDetection
    },
    {
        "house-number-detection",
        maps::mrc::db::toloka::TaskType::HouseNumberDetection
    },
    {
        "house-number-recognition",
        maps::mrc::db::toloka::TaskType::HouseNumberRecognition
    },
    {
        "detection-missing-on-frame",
        maps::mrc::db::toloka::TaskType::DetectionMissingOnFrame
    },
};

std::string toString(const maps::json::Value& value) {
    std::stringstream ss;

    maps::json::Builder builder(ss);
    ss << value;

    return ss.str();
}

} // namespace

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser;

    auto mrcConfigPath = parser.string("mrc-config")
            .help("path to mrc config");

    auto platformOpt = parser.string("platform")
        .required()
        .help("Platform to operate with ('toloka' or 'yang')");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto taskTypeName = parser.string("task-type")
            .help("name of task type in toloka");

    auto goldenTasksPath = parser.string("golden-tasks")
            .help("path to json file with golden tasks");

    parser.parse(argc, argv);

    auto platform =
        maps::enum_io::fromString<maps::mrc::db::toloka::Platform>(platformOpt);

    const auto mrcConfig = maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);
    auto poolHolder = mrcConfig.makePoolHolder();

    const auto taskType = TASK_TYPE_NAME_TO_TYPE.at(taskTypeName);

    maps::json::Value goldenTasksJson = maps::json::Value::fromFile(goldenTasksPath);

    maps::mrc::db::toloka::Tasks tasks;

    for (const auto& goldenTaskJson : goldenTasksJson) {
        maps::mrc::db::toloka::Task task(platform);

        task.setType(taskType)
            .setStatus(maps::mrc::db::toloka::TaskStatus::InProgress)
            .setInputValues(toString(goldenTaskJson["inputValues"]))
            .setKnownSolutions(toString(goldenTaskJson["knownSolutions"]));

        tasks.push_back(task);
    }

    auto txn = poolHolder.pool().masterWriteableTransaction();
    maps::mrc::db::toloka::TaskGateway(*txn).insertx(tasks);
    txn->commit();

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    ERROR() << "unknown error";
    return EXIT_FAILURE;
}
