#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <opencv2/opencv.hpp>

#include <string>
#include <vector>

namespace {

class PanelDetectorBgnd {
public:
    PanelDetectorBgnd()
        : panelEMA(-1.f)
    {}

    float getPanelEMACoord() const {
        return panelEMA;
    }

    int update(const cv::Mat &img) {
        constexpr float EMAkoef = 0.1f;

        cv::Mat curGradMagn; gradientMagn(img, curGradMagn);
        if (prevGradMagn.empty()) {
            curGradMagn.copyTo(prevGradMagn);
            return -1;
        }
        cv::Mat absDiff; cv::absdiff(prevGradMagn, curGradMagn, absDiff);

        int curVertDelimeter = bgndVertDelimeter(absDiff);
        if (-1 != curVertDelimeter) {
            if (panelEMA < 0.f)
                panelEMA = (float)curVertDelimeter;
            else
                panelEMA = (1.f - EMAkoef) * panelEMA + EMAkoef * (float)curVertDelimeter;
        }
        curGradMagn.copyTo(prevGradMagn);
        return curVertDelimeter;
    }
private:
    cv::Mat prevGradMagn;
    float panelEMA;

    static void gradientMagn(const cv::Mat &img, cv::Mat &gradMagn) {
        CV_Assert((img.type() == CV_8UC1) || (img.type() == CV_8UC3));
        cv::Mat gray;
        if (img.type() == CV_8UC3)
            cv::cvtColor(img, gray, cv::COLOR_BGR2GRAY);
        else
            img.copyTo(gray);
        cv::GaussianBlur(gray, gray, cv::Size(), 0.9, 0.9);

        cv::Mat gradX, gradY;
        cv::Sobel(gray, gradX, CV_32F, 1, 0);
        cv::Sobel(gray, gradY, CV_32F, 1, 0);
        cv::sqrt(gradX.mul(gradX) + gradY.mul(gradY), gradMagn);
    }

    static int bgndVertDelimeter(const cv::Mat &data) {
        constexpr int bgndVertMinPart = 6;
        constexpr float bgndThrCoeff = 5.f;

        cv::Mat rowHist;
        cv::reduce(data, rowHist, 1, cv::REDUCE_SUM, CV_32F);

        const float bgndthr = bgndThrCoeff * data.cols;

        int row = rowHist.rows - 1;
        for (; 0 <= row; row--) {
            if (rowHist.at<float>(row) < bgndthr)
                break;
        }
        if ((rowHist.rows - row) * bgndVertMinPart > rowHist.rows)
            row = rowHist.rows - 1;

        for (; 0 <= row; row--) {
            if (rowHist.at<float>(row) > bgndthr)
                break;
        }
        if ((rowHist.rows - row) * bgndVertMinPart > rowHist.rows)
            return -1;
        return row;
    }
};

class VanishingPointDetector {
public:
    VanishingPointDetector() {}

    cv::Point2f detect(const std::vector<cv::Vec4i> &lines, const cv::Size &imgSize, float &weight) {
        constexpr int denom = 5;
        constexpr double gausSigma = 1.5;
        constexpr int maxLocAreaSize = 5;
        cv::Point2f inters;
        cv::Mat conf = cv::Mat::zeros(imgSize / denom, CV_32FC1);
        for (size_t i = 0; i < lines.size() - 1; i++) {
            const cv::Vec4i &line1 = lines[i];
            if (!isLineValid(line1))
                continue;
            if (std::min(line1[1], line1[3]) < imgSize.height / 2)
                continue;

            const float len1 = sqrtf((float)(line1[0] - line1[2]) * (line1[0] - line1[2]) + (float)(line1[1] - line1[3]) * (line1[1] - line1[3]));
            for (size_t j = i + 1; j < lines.size(); j++) {
                const cv::Vec4i &line2 = lines[j];
                if (!isLineValid(line2))
                    continue;
                if (std::min(line2[1], line2[3]) < imgSize.height / 2)
                    continue;

                if (!lineIntersection(line1, lines[j], inters))
                    continue;

                if (inters.x > 2 * imgSize.width / 3 || inters.x < imgSize.width / 3)
                    continue;

                if (inters.y > std::min(line1[1], line1[3]) ||
                    inters.y > std::min(line2[1], line2[3]))
                    continue;

                int icol = (int)(inters.x / denom);
                int irow = (int)(inters.y / denom);
                if (icol < 0 || icol + 1 >= conf.cols ||
                    irow < 0 || irow + 1 >= conf.rows)
                    continue;
                const float len2 = sqrtf((float)(line2[0] - line2[2]) * (line2[0] - line2[2]) + (float)(line2[1] - line2[3]) * (line2[1] - line2[3]));
                const float koef = len1 * len2;
                conf.at<float>(irow, icol) += koef;
            }
        }

        cv::Mat conf_blured;
        cv::GaussianBlur(conf, conf_blured, cv::Size(), gausSigma);

        double minVal, maxVal;
        cv::Point maxLoc;
        cv::minMaxLoc(conf_blured, &minVal, &maxVal, 0, &maxLoc);

        cv::Rect maxLocArea;
        maxLocArea.x = std::max(maxLoc.x - maxLocAreaSize, 0);
        maxLocArea.y = std::max(maxLoc.y - maxLocAreaSize, 0);
        maxLocArea.width  = std::min(maxLoc.x + maxLocAreaSize, conf_blured.cols - 1) - maxLocArea.x + 1;
        maxLocArea.height = std::min(maxLoc.y + maxLocAreaSize, conf_blured.rows - 1) - maxLocArea.y + 1;

        conf_blured(maxLocArea).setTo(0.);

        double minValSec, maxValSec;
        cv::Point maxLocSec;
        cv::minMaxLoc(conf_blured, &minValSec, &maxValSec, 0, &maxLocSec);
        weight = (float)((maxVal - maxValSec) / maxVal);
        return maxLoc * denom;
    }
private:
    static bool isLineValid(const cv::Vec4i &line) {
        constexpr float deltaAngle = 10.f * (float)CV_PI / 180.f;
        if (line[2] == line[0] || line[3] == line[1])
            return false;
        if (line[2] != line[0] && fabs(atanf((float)(line[3] - line[1]) / (float)(line[2] - line[0]))) < deltaAngle)
            return false;
        return true;
    }

    static bool lineIntersection(const cv::Vec4i &line1, const cv::Vec4i &line2, cv::Point2f &intersection) {
        int denom = ((line1[0] - line1[2]) * (line2[1] - line2[3]) - (line1[1] - line1[3]) * (line2[0] - line2[2]));
        if (0 == denom)
            return false;
        intersection.x = (float)((line1[0] * line1[3] - line1[1] * line1[2]) * (line2[0] - line2[2]) -
            (line2[0] * line2[3] - line2[1] * line2[2]) * (line1[0] - line1[2])) / denom;
        intersection.y = (float)((line1[0] * line1[3] - line1[1] * line1[2]) * (line2[1] - line2[3]) -
            (line2[0] * line2[3] - line2[1] * line2[2]) * (line1[1] - line1[3])) / denom;
        return true;
    }
};

static void calcViewAreaByVP(const cv::Point2f &pt, int img_height, std::vector<cv::Point2f> &pts) {
    constexpr float vpAngleWidthDegree = 75.f;
    constexpr float angle = vpAngleWidthDegree * (float)CV_PI / 180.f;
    constexpr float horzSkipPart = 1.f / 3.f;

    float height = (float)img_height - pt.y;
    pts.resize(4);
    pts[0].x = pt.x - tanf(angle) * height * horzSkipPart;
    pts[0].y = pt.y + height * horzSkipPart;

    pts[1].x = pt.x - tanf(angle) * height;
    pts[1].y = (float)img_height;

    pts[2].x = pt.x + tanf(angle) * height;
    pts[2].y = (float)img_height;

    pts[3].x = pt.x + tanf(angle) * height * horzSkipPart;
    pts[3].y = pt.y + height * horzSkipPart;
}

static cv::Mat findHomography(const std::vector<cv::Point2f> &pts, const cv::Size &birdview_size){
    std::vector<cv::Point2f> pts2(4);
    pts2[0].x = 0;
    pts2[0].y = 0;

    pts2[1].x = 0;
    pts2[1].y = (float)birdview_size.height;

    pts2[2].x = (float)birdview_size.width;
    pts2[2].y = (float)birdview_size.height;

    pts2[3].x = (float)birdview_size.width;
    pts2[3].y = 0;

    cv::Mat H = cv::findHomography(pts, pts2);
    H.convertTo(H, CV_32F);
    return H;
}

};

int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;
    auto inputFilterParam = parser.string("input-filter")
        .required()
        .help("filter path for input images");

    auto outputFolderPathParam = parser.string("output-folder")
        .required()
        .help("path to output folder");

    parser.parse(argc, argv);

    std::vector<cv::String> files;
    cv::glob(inputFilterParam, files);

    REQUIRE(1 <= files.size(),
            "There are no images found by filter");

    PanelDetectorBgnd panelDetBgnd;
    VanishingPointDetector vanPtDet;

    for (size_t i = 0; i < files.size(); i++) {
        INFO() << "Rework file: " << files[i];

        cv::Mat img = cv::imread(files[i], cv::IMREAD_COLOR);
        cv::Mat gray; cv::cvtColor(img, gray, cv::COLOR_BGR2GRAY);
        panelDetBgnd.update(gray);

        const float panelDetBgndEMA = panelDetBgnd.getPanelEMACoord();
        if (panelDetBgndEMA < 0.f) {
            continue;
        }

        std::vector<cv::Vec4i> lines;
        cv::Ptr<cv::LineSegmentDetector> det = cv::createLineSegmentDetector(cv::LSD_REFINE_STD, 0.5);
        det->detect(gray.rowRange(0, (int)panelDetBgndEMA + 1), lines);

        float vanPtWeight;
        const cv::Point2f pt = vanPtDet.detect(lines, cv::Size(gray.cols, (int)panelDetBgndEMA + 1), vanPtWeight);

        std::vector<cv::Point2f> pts;
        calcViewAreaByVP(pt, (int)panelDetBgndEMA, pts);
        const int varOutSize = (int)(pts[2].x - pts[1].x);

        cv::Mat H = findHomography(pts, cv::Size(varOutSize, varOutSize));

        cv::Mat birdview;
        cv::warpPerspective(img, birdview, H, cv::Size(varOutSize, varOutSize));

        if (vanPtWeight < 0.3f)
            cv::putText(birdview, "BAD", cv::Point(10, 30), 0, 1., cv::Scalar(0, 0, 255), 2);
        else if (vanPtWeight < 0.5f)
            cv::putText(birdview, "MIDDLE", cv::Point(10, 30), 0, 1., cv::Scalar(0, 255, 255), 2);
        else
            cv::putText(birdview, "GOOD", cv::Point(10, 30), 0, 1., cv::Scalar(0, 255, 0), 2);

        size_t pos = files[i].find_last_of("/\\");
        std::string filename = files[i];
        if (std::string::npos != pos) {
            filename = files[i].substr(pos);
        }
        filename = (std::string)outputFolderPathParam + "/" + filename;
        cv::imwrite(filename, birdview);
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
