#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/exif.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/services/mrc/libs/camera/include/camera_parameters.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <opencv2/opencv.hpp>

#include <string>
#include <vector>

namespace common = maps::mrc::common;


int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;
    auto inputFilterParam = parser.string("input-filter")
        .required()
        .help("filter path for input images");

    auto outputFolderPathParam = parser.string("output-folder")
        .required()
        .help("path to output folder");

    auto realCameraHeightParam = parser.real("real-camera-height")
             .defaultValue(1.)
             .help("height of the real camera fixing point");

    auto realCameraAngleParam = parser.real("real-camera-angle")
             .defaultValue(0.)
             .help("angle of the real camera optical axis to the road in degree (default: 0 - the camera is parallel to a road, positive - camera rotate to the road)");

    auto birdCameraHeightParam = parser.real("bird-camera-height")
             .defaultValue(23.)
             .help("height of the bird camera");

    auto birdCameraDistParam = parser.real("bird-camera-distance")
             .defaultValue(48.)
             .help("distance of the bird camera from the real camera in car moving direction");


    parser.parse(argc, argv);

    std::vector<cv::String> files;
    cv::glob(inputFilterParam, files);

    REQUIRE(1 <= files.size(),
            "There are no images found by filter");

    cv::Size outFrameSize = cv::imread(files[0]).size();
    const maps::mrc::CameraParameters parameters = maps::mrc::getCameraParameters(
        "unknown model",
        common::ImageOrientation(false, common::Rotation::CW_0),
        outFrameSize);
    outFrameSize.height *= 2;


    cv::Mat cameraIntrinsic;
    parameters.cameraMatrix.convertTo(cameraIntrinsic, CV_32F);

    const float d = realCameraHeightParam;

    const float alphaDegree = 90.f - realCameraAngleParam;
    const float dy = birdCameraHeightParam;
    const float dz = birdCameraDistParam;

    const float cs = cos(alphaDegree * CV_PI / 180.);
    const float ss = sin(alphaDegree * CV_PI / 180.);
    cv::Mat Ri = cv::Mat::eye(3, 3, CV_32FC1);
    Ri.at<float>(0, 0) = 1.f; Ri.at<float>(0, 1) = 0.f; Ri.at<float>(0, 2) = 0.f;
    Ri.at<float>(1, 0) = 0.f; Ri.at<float>(1, 1) = cs; Ri.at<float>(1, 2) = d;
    Ri.at<float>(2, 0) = 0.f; Ri.at<float>(2, 1) = ss; Ri.at<float>(2, 2) = 0.f;
    cv::Mat Ro = cv::Mat::eye(3, 3, CV_32FC1);
    Ro.at<float>(0, 0) = 1.f; Ro.at<float>(0, 1) = 0.f; Ro.at<float>(0, 2) = 0.f;
    Ro.at<float>(1, 0) = 0.f; Ro.at<float>(1, 1) = -1.f; Ro.at<float>(1, 2) = dy;
    Ro.at<float>(2, 0) = 0.f; Ro.at<float>(2, 1) = 0.f; Ro.at<float>(2, 2) = dz;

    const cv::Mat H = cameraIntrinsic * Ro * Ri.inv() * cameraIntrinsic.inv();
    cv::Mat outFrame(outFrameSize, CV_8UC3);
    cv::Mat birdeye = outFrame.rowRange(0, outFrame.rows / 2);
    cv::Mat img = outFrame.rowRange(outFrame.rows / 2, outFrame.rows);

    for (size_t i = 0; i < files.size(); i++) {
        INFO() << "Rework file: " << files[i];
        cv::imread(files[i], cv::IMREAD_COLOR).copyTo(img);
        cv::warpPerspective(img, birdeye, H, img.size());
        size_t pos = files[i].find_last_of("/\\");
        std::string filename = files[i];
        if (std::string::npos != pos) {
            filename = files[i].substr(pos);
        }
        filename = (std::string)outputFolderPathParam + "/" + filename;
        cv::imwrite(filename, outFrame);
    }

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
