#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <maps/libs/common/include/file_utils.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <opencv2/opencv.hpp>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>


using namespace maps::mrc;

namespace {

std::vector<db::TId> loadAssignmentIds(const std::string& path) {
    std::vector<db::TId> results;
    std::ifstream ifs(path);
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        if ('#' == line[0])
            continue;
        results.push_back(std::stoi(line));
    }
    return results;
}

std::set<db::TId> loadIds(const std::string& path) {
    std::set<db::TId> results;
    std::ifstream ifs(path);
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        if ('#' == line[0])
            continue;
        results.insert(std::stoi(line));
    }
    return results;
}

db::Features loadAssignmentPhotos(maps::pgpool3::Pool& pool, db::TId assignmentId)
{
    const auto CLASSIFICATION_FILTER
    = db::table::Feature::quality.isNotNull()
      && db::table::Feature::roadProbability.isNotNull()
      && db::table::Feature::forbiddenProbability.isNotNull();

    auto txn = pool.slaveTransaction();
    return db::FeatureGateway{*txn}.load(
        db::table::Feature::assignmentId.equals(assignmentId)
        && CLASSIFICATION_FILTER);
}

bool needSplit(const db::Feature& lhs, const db::Feature& rhs) {
//    constexpr auto TIME_GAP = std::chrono::seconds{60};
    constexpr auto TIME_GAP = std::chrono::seconds{300};
    return lhs.sourceId() != rhs.sourceId() || std::chrono::abs(lhs.timestamp() - rhs.timestamp()) > TIME_GAP;
}

db::Features loadFeatures(maps::pgpool3::Pool& pool, const std::string& sourceId, const maps::chrono::TimePoint& start, const maps::chrono::TimePoint& end)
{
    auto txn = pool.slaveTransaction();
    return db::FeatureGateway{*txn}.load(
        db::table::Feature::sourceId.equals(sourceId) &&
        db::table::Feature::isPublished &&
        db::table::Feature::date.between(start, end)
    );
}

template <class FeatureIt>
void savePairs(std::ofstream& ofs, FeatureIt first, FeatureIt last, const std::set<db::TId>& filterIds) {
    first++;
    if (first == last)
        return;
    if (filterIds.empty()) {
        for (FeatureIt it = first; it != last; it++) {
            const db::Feature& f1 = *std::prev(it);
            const db::Feature& f2 = *it;
            ofs << "http://storage-int.mds.yandex.net/get-maps_mrc/" << f1.mdsGroupId() << "/" << f1.mdsPath() << " "
                << (f1.hasOrientation() ? (int)f1.orientation() : (int)1) << " "
                << "http://storage-int.mds.yandex.net/get-maps_mrc/" << f2.mdsGroupId() << "/" << f2.mdsPath() << " "
                << (f2.hasOrientation() ? (int)f2.orientation() : (int)1) << " "
                << ((f1.cameraDeviation() == db::CameraDeviation::Front) ? "0" : "1")
                << std::endl;
        }
    } else {
        for (FeatureIt it = first; it != last; it++) {
            const db::Feature& f1 = *std::prev(it);
            const db::Feature& f2 = *it;
            if (0 < filterIds.count(f1.id())) {
                INFO() << "Filter feature id: " << f1.id();
                continue;
            }
            if (0 < filterIds.count(f2.id())) {
                INFO() << "Filter feature id: " << f2.id();
                continue;
            }
            ofs << "http://storage-int.mds.yandex.net/get-maps_mrc/" << f1.mdsGroupId() << "/" << f1.mdsPath() << " "
                << (f1.hasOrientation() ? (int)f1.orientation() : (int)1) << " "
                << "http://storage-int.mds.yandex.net/get-maps_mrc/" << f2.mdsGroupId() << "/" << f2.mdsPath() << " "
                << (f2.hasOrientation() ? (int)f2.orientation() : (int)1) << " "
                << ((f1.cameraDeviation() == db::CameraDeviation::Front) ? "0" : "1")
                << std::endl;
        }
    }
}

} //namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Extract pairs of consecutive features ");

    maps::cmdline::Option<std::string> inputPath = parser.string("input")
        .help("Path to input file with assignment ids");

    maps::cmdline::Option<std::string> sourceId = parser.string("source-id")
        .help("Source ID");

    maps::cmdline::Option<std::string> timeStart = parser.string("time-start")
        .help("Start time stamp (format: 2000-08-01 00:00:00.000000+03)");

    maps::cmdline::Option<std::string> timeEnd = parser.string("time-end")
        .help("End time stamp (format: 2000-08-01 00:00:00.000000+03)");

    maps::cmdline::Option<std::string> outputPath = parser.string("output")
        .required()
        .help("Path to output file with features");

    maps::cmdline::Option<std::string> filterPath = parser.string("filter")
        .help("Path to file with features, which must be filter");

    maps::cmdline::Option<std::string> mrcConfigPath = parser.string("mrc-config")
        .help("Path to mrc config");

    maps::cmdline::Option<std::string> secretVersion = parser.string("secret-version")
        .help("version for secrets from yav.yandex-team.ru");

    parser.parse(argc, const_cast<char**>(argv));

    maps::mrc::common::Config mrcConfig =
        secretVersion.defined()
        ? maps::mrc::common::Config(maps::vault_boy::loadContextWithYaVault(secretVersion), mrcConfigPath)
        : maps::mrc::common::templateConfigFromCmdPath(mrcConfigPath);
    maps::wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());

    if (!inputPath.empty()) {
        std::vector<db::TId> assignmentIds = loadAssignmentIds(inputPath);
        std::set<db::TId> filterIds = (filterPath.defined() ? loadIds(filterPath) : std::set<db::TId>());

        std::ofstream ofs(outputPath);
        for (size_t i = 0; i < assignmentIds.size(); i++) {
            db::TId assignmentId = assignmentIds[i];
            db::Features features = loadAssignmentPhotos(mrc.pool(), assignmentId);
            std::sort(features.begin(), features.end(),
                      [](const auto& lhs, const auto& rhs) {
                          return std::make_tuple(lhs.sourceId(), lhs.timestamp())
                                 < std::make_tuple(rhs.sourceId(), rhs.timestamp());
                      });
            INFO() << assignmentId << " " << features.size();
            auto first = features.begin();
            auto itprev = first;
            for (auto it = first + 1; it != features.end(); it++) {
                if (needSplit(*itprev, *it)) {
                    savePairs(ofs, first, it, filterIds);
                    first = it;
                }
                itprev = it;
            }
            if (first != features.end())
                savePairs(ofs, first, features.end(), filterIds);
        }
    } else if (!sourceId.empty()) {
        std::ofstream ofs(outputPath);
        db::Features features = loadFeatures(mrc.pool(), sourceId, maps::chrono::parseSqlDateTime(timeStart), maps::chrono::parseSqlDateTime(timeEnd));
        INFO() << "Loaded: " << features.size() << " features";
        std::sort(features.begin(), features.end(),
            [](const db::Feature& a, const db::Feature& b) {
                return a.timestamp() < b.timestamp();
            }
        );
        savePairs(ofs, features.begin(), features.end(), {});
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
