#include "context.h"

#include <maps/libs/nirvana/include/client.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/for_each_batch.h>

#include <random>

namespace maps::mrc::tools {
namespace {

const std::string APP_NAME = "collect_traffic_signs";

auto findOperationByName(nirvana::Workflow& work, const std::string& name)
{
    auto it = std::find_if(work.operations().begin(), work.operations().end(),
                           [&](const auto& op) { return op.name() == name; });
    REQUIRE(it != work.operations().end(),
            "Operation " << name << " does not exists in Nirvana workflow "
                         << work.globalId());
    return *it;
}

} // anonymous namespace

Context::Context(common::Config& cfg, std::string nirvanaToken)
    : cfg_{cfg}, nirvanaToken_{std::move(nirvanaToken)}
{
}

nirvana::OperationParameters
Context::postToPublicMds(const std::string& nirvanaWorkflow,
                         const std::string& inputJson)
{
    auto publicMds = cfg_.makePublicMdsClient();
    std::ostringstream path;
    path << APP_NAME << "/" << nirvanaWorkflow;
    auto resp = publicMds.post(path.str(), inputJson);
    auto url = publicMds.makeReadUrl(resp.key());
    INFO() << "Posted: " << url;
    return {{"url", json::Value::fromString('"' + url + '"')}};
}

Photos Context::postToPublicMds(const db::TIds& featureIds)
{
    static std::mt19937 gen{std::random_device{}()};
    static std::uniform_int_distribution<short> dist{0, 10000};

    Photos result;
    auto postgres = cfg_.makePoolHolder();
    auto mds = cfg_.makeMdsClient();
    auto publicMds = cfg_.makePublicMdsClient();
    try {
        constexpr size_t BATCH_SIZE = 100;
        common::forEachBatch(
            featureIds, BATCH_SIZE, [&](auto first, auto last) {
                auto features
                    = db::FeatureGateway{*postgres.pool().slaveTransaction()}
                          .load(db::table::Feature::id.in({first, last}));
                for (const auto& feature : features) {
                    auto orient = feature.orientation();
                    auto img = mds.get(feature.mdsKey());
                    img = common::transformByImageOrientation(img, orient);
                    std::ostringstream path;
                    path << APP_NAME << "/" << std::setfill('0')
                         << std::setw(4) << dist(gen) << "/" << feature.id();
                    auto resp = publicMds.post(path.str(), img);
                    auto url = publicMds.makeReadUrl(resp.key());
                    result.push_back({feature.id(), resp.key(), url});
                    INFO() << "Posted: " << url;
                }
            });
        return result;
    }
    catch (...) {
        deleteFromPublicMds(result);
        throw;
    }
}

void Context::deleteFromPublicMds(const Photos& photos)
{
    auto publicMds = cfg_.makePublicMdsClient();
    for (const auto& photo : photos) {
        publicMds.del(photo.publicMdsKey);
        INFO() << "Deleted: " << photo.url;
    }
}

void Context::runWorkflow(const std::string& workflow,
                          const std::string& inputJson,
                          const std::string& outputPath)
{
    static const auto INPUT_OP = "sh wget (one file)";
    static const auto OUTPUT_OP = "upload_traffic_signs_dataset_to_yt";

    INFO() << "Connecting to Nirvana: " << nirvanaToken_;
    auto client = nirvana::Client{
        APP_NAME, std::make_unique<nirvana::OAuth>(nirvanaToken_)};
    INFO() << "Getting workflow: " << workflow;
    auto original = client.getWorkflow(workflow);
    INFO() << "Cloning workflow: " << original.globalId();
    auto clone = client.cloneWorkflow(original);
    INFO() << "Setting parameters";
    client.setOperationParameters(
        clone, findOperationByName(clone, INPUT_OP),
        postToPublicMds(clone.globalId(), inputJson));
    client.setOperationParameters(
        clone, findOperationByName(clone, OUTPUT_OP),
        {{"output", json::Value::fromString('"' + outputPath + '"')}});
    INFO() << "Starting workflow: " << clone.globalId();
    client.startWorkflow(clone);
    INFO() << "Workflow URL: " << clone.userInterfaceUrl().toString();
}

} // maps::mrc::tools
