#include <maps/wikimap/mapspro/services/mrc/tools/collect_traffic_signs/merge_classification_operation/tools.h>

#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/json/include/prettify.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/opencv.h>
#include <maps/libs/geolib/include/vector.h>
#include <yandex/maps/mrc/unittest/local_server.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

namespace maps::mrc::tools::test {
namespace {

const auto IMAGE = common::getTestImage<std::string>();

const auto DATA_DIR = ArcadiaSourceRoot() + "/maps/wikimap/mapspro/services/"
                                            "mrc/tools/collect_traffic_signs/"
                                            "merge_classification_operation/"
                                            "tests";

const auto INPUT_JSON
    = maps::common::readFileToString(DATA_DIR + "/input.json");

const auto OUTPUT_JSON
    = maps::common::readFileToString(DATA_DIR + "/output.json");

const std::string URLS[]
    = {"http://storage.mds.yandex.net:80/get-maps_mrc_public/222994/"
       "toloka-mgr/task/6042363"};

void replaceAll(std::string& str,
                const std::string& from,
                const std::string& to)
{
    size_t start_pos = 0;
    while ((start_pos = str.find(from, start_pos)) != std::string::npos) {
        str.replace(start_pos, from.length(), to);
        start_pos += to.length();
    }
}

std::string adjustUrls(const std::string& json,
                       const std::vector<std::string>& urls)
{
    EXPECT_EQ(std::size(URLS), urls.size());
    auto result = json;
    for (size_t i = 0; i < std::size(URLS); ++i) {
        replaceAll(result, URLS[i], urls[i]);
    }
    return result;
}

inline geolib3::BoundingBox bbox(double x1, double y1, double x2, double y2)
{
    return {geolib3::Point2(x1, y1), geolib3::Point2(x2, y2)};
}

const BboxToSignMap::value_type EXPECTED_BBOX_TO_SIGNS[]
    = {{bbox(1209, 363, 1267, 403),
        traffic_signs::TrafficSign::InformationTowTruck},
       {bbox(1214, 223, 1284, 282),
        traffic_signs::TrafficSign::PrescriptionPedestrianCrossing},
       {bbox(1214, 296, 1264, 356),
        traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping},
       {bbox(128, 410, 169, 450),
        traffic_signs::TrafficSign::ProhibitoryMaxHeight},
       {bbox(784, 398, 817, 433),
        traffic_signs::TrafficSign::PrescriptionPedestrianCrossing}};

} // anonymous namespace

struct Test : testing::Test {
    using TestFixture
        = unittest::WithUnittestConfig<unittest::MdsStubFixture>;

    Test() { testFixture_.reset(new TestFixture); }


    std::unique_ptr<TestFixture> testFixture_;
};

    TEST_F(Test, test)
    {
        auto cfg = testFixture_->config();
        auto mds = testFixture_->config().makePublicMdsClient();
        std::vector<std::string> urls;
        for (size_t id = 1; id <= std::size(URLS); ++id) {
            auto res = mds.post("tst/" + std::to_string(id), IMAGE);
            urls.push_back(mds.makeReadUrl(res.key()));
        }
        auto inputJson
            = json::Value::fromString(adjustUrls(INPUT_JSON, urls));

        auto urlToBase64DataMap = loadPhotos(inputJson);
        EXPECT_EQ(urlToBase64DataMap.size(), 1u);

        auto urlToSignsMap = aggregateSigns(inputJson);
        EXPECT_EQ(urlToSignsMap.size(), 1u);
        auto bboxToSignMap = urlToSignsMap.begin()->second;
        EXPECT_TRUE(
            std::is_permutation(bboxToSignMap.begin(), bboxToSignMap.end(),
                                std::begin(EXPECTED_BBOX_TO_SIGNS),
                                std::end(EXPECTED_BBOX_TO_SIGNS),
                                [](const auto& lhs, const auto& rhs) {
                                    return BboxEqualTo{}(lhs.first, rhs.first)
                                           && lhs.second == rhs.second;
                                }));

        std::ostringstream os;
        dump(urlToBase64DataMap, urlToSignsMap, os);
        EXPECT_EQ(json::Value::fromString(os.str()),
                  json::Value::fromString(OUTPUT_JSON));
    }

} // maps::mrc::tools::test
