#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/temp_table.h>

#include <string>
#include <fstream>


namespace {
struct Data {
    double lon;
    double lat;
    maps::chrono::TimePoint timestamp;
};
}

int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("Export couriers tracks from YT table to json");

    maps::cmdline::Option<std::string> intputTableName = parser.string("input")
        .defaultValue("//home/logfeller/logs/taxi-yagr-courier-positions-log/1d/2020-06-27")
        .help("Input YT table name");

    maps::cmdline::Option<std::string> id2typeTableName = parser.string("id2type")
        .defaultValue("//home/maps/core/mrc/vbystricky/eda_courier_id2type")
        .help("YT table with id to type map");

    maps::cmdline::Option<std::string> outputJsonPath = parser.string("output")
        .defaultValue("./output.json")
        .help("Path to output json file");

    maps::cmdline::Option<int> recordsCnt = parser.num("count")
        .defaultValue(1000);

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Connecting to yt::hahn";
    NYT::IClientPtr client = NYT::CreateClient("hahn");

    std::map<int, std::string> id2type;
    {
        NYT::TTableReaderPtr<NYT::TNode> reader = client->CreateTableReader<NYT::TNode>(id2typeTableName.c_str());
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& inpRow = reader->GetRow();
            const int id = inpRow["id"].AsInt64();
            id2type[id] = inpRow["type"].AsString().c_str();
        }
    }

    const NYT::TTempTable sortedTable(client);

    client->Sort(
        NYT::TSortOperationSpec()
            .AddInput(intputTableName.c_str())
            .Output(sortedTable.Name())
            .SortBy({"courier_id", "date"})
    );

    NYT::TTableReaderPtr<NYT::TNode> reader = client->CreateTableReader<NYT::TNode>(sortedTable.Name());
    std::ofstream ofs(outputJsonPath.c_str());
    REQUIRE(ofs.is_open(), "Unable to open output file: " << outputJsonPath);

    ofs << "[" << std::endl;
    int restCouriers = recordsCnt;
    int courierId = -1;
    int fakeGPSCount = 0;
    std::vector<Data> courierData;
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& inpRow = reader->GetRow();
        int newCourierId = atoi(inpRow["courier_id"].AsString().c_str());
        if (-1 == courierId) {
            courierId = newCourierId;
            fakeGPSCount = 0;
        } else if (courierId != newCourierId) {
            if (10 * fakeGPSCount < (int)courierData.size()) {
                std::sort(courierData.begin(), courierData.end(),
                    [](const Data& lhs, const Data& rhs) {
                        return lhs.timestamp < rhs.timestamp;
                    }
                );
                restCouriers--;
                ofs << "    {" << std::endl;
                ofs << "        \"courier_id\" : " << courierId << "," << std::endl;
                ofs << "        \"type\" : \"" << id2type[courierId] << "\"," << std::endl;
                ofs << "        \"track\" : " << "[" << std::endl;
                for (size_t i = 0; i < courierData.size() - 1; i++) {
                    const Data& d = courierData[i];
                    ofs << "            {" << std::endl;
                    ofs << "                \"timestamp\" : \"" << maps::chrono::formatSqlDateTime(d.timestamp) << "\"," << std::endl;
                    ofs << "                \"lon\" : " << d.lon << "," << std::endl;
                    ofs << "                \"lat\" : " << d.lat << std::endl;
                    ofs << "            }," << std::endl;
                }
                    const Data& d = courierData[courierData.size() - 1];
                ofs << "            {" << std::endl;
                ofs << "                \"timestamp\" : \"" << maps::chrono::formatSqlDateTime(d.timestamp) << "\"," << std::endl;
                ofs << "                \"lon\" : " << d.lon << "," << std::endl;
                ofs << "                \"lat\" : " << d.lat << std::endl;
                ofs << "            }" << std::endl;

                ofs << "        ]" << std::endl;
                if (0 < restCouriers) {
                    ofs << "    }," << std::endl;
                } else {
                    ofs << "    }" << std::endl;
                    break;
                }

                if (0 == (restCouriers % 100)) {
                    INFO() << "Rest " << restCouriers << " couriers for reworked";
                }
            }
            courierData.clear();
            courierId = newCourierId;
            fakeGPSCount = 0;
        }
        if (inpRow["fakeGPS"].AsBool()) {
            fakeGPSCount++;
        } else {
            Data data;
            data.lon = inpRow["longitude"].AsDouble();
            data.lat = inpRow["latitude"].AsDouble();
            data.timestamp = maps::chrono::parseIsoDateTime(inpRow["timestamp"].AsString().c_str());
            courierData.emplace_back(data);
        }
    }

    ofs << "]" << std::endl;
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
