#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/common/include/retry.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/http/include/http.h>
#include <yandex/maps/mrc/traffic_signs/signs.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/conversion.h>
#include <mapreduce/yt/interface/client.h>
#include <util/system/datetime.h>
#include <util/random/random.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>

#include <string>
#include <vector>
#include <regex>

using namespace maps;
using namespace maps::mrc;

namespace {

static const std::string JSON_NAME_IMAGES_SET   = "images_with_objects";
static const std::string JSON_NAME_FEATURE_ID   = "feature_id";
static const std::string JSON_NAME_IMAGE        = "image";
static const std::string JSON_NAME_OBJECTS      = "objects";
static const std::string JSON_NAME_LAT          = "lat";
static const std::string JSON_NAME_LON          = "lon";


db::TIds loadFeatureIds(const std::string& fileName) {
    db::TIds featureIds;

    std::ifstream ifs(fileName);
    REQUIRE(ifs.is_open(), "Failed to open file " << fileName);
    while (!ifs.eof()) {
        std::string line;
        std::getline(ifs, line);

        if (line.empty()) {
            continue;
        }

        featureIds.push_back(atoll(line.c_str()));
    }

    return featureIds;
}

db::Features loadFeatures(pgpool3::Pool& pool, const db::TIds& featureIds) {
    auto txn = pool.masterReadOnlyTransaction();
    return db::FeatureGateway(*txn).loadByIds(featureIds);
}

std::vector<uint8_t> downloadImage(maps::http::Client& client, const std::string& url)
{
    maps::common::RetryPolicy retryPolicy;
    retryPolicy.setTryNumber(10)
        .setInitialCooldown(std::chrono::seconds(1))
        .setCooldownBackoff(2);

    auto validateResponse = [](const auto& maybeResponse) {
        return maybeResponse.valid() && maybeResponse.get().responseClass() != maps::http::ResponseClass::ServerError;
    };
    auto resp = maps::common::retry(
                [&]() {
                    return maps::http::Request(client, maps::http::GET, maps::http::URL(url)).perform();
                },
                retryPolicy,
                validateResponse
            );
    REQUIRE(resp.responseClass() == maps::http::ResponseClass::Success,
        "Unexpected response status " << resp.status() << " for url "
        << url);
    return resp.readBodyToVector();
}

cv::Mat loadImage(
    maps::http::Client& client,
    const std::string& url,
    maps::mrc::common::ImageOrientation orientation)
{
    std::vector<uint8_t> data = downloadImage(client, url);
    cv::Mat image = cv::imdecode(data, cv::IMREAD_COLOR | cv::IMREAD_IGNORE_ORIENTATION);
    return maps::mrc::common::transformByImageOrientation(image, orientation);
}

std::string getFeatureUrl(const maps::mrc::db::Feature& f) {
    return "http://storage-int.mds.yandex.net/get-maps_mrc/" + f.mdsGroupId() + "/" + f.mdsPath();
}

cv::Mat loadFeatureImage(
    maps::http::Client& client,
    const maps::mrc::db::Feature& f)
{
    return loadImage(client, getFeatureUrl(f), f.orientation());
}

std::string encodeImage(const cv::Mat& image) {
    std::vector<uint8_t> encImage;
    cv::imencode(".jpg", image, encImage);
    return maps::base64Encode(std::string(encImage.begin(), encImage.end()));
}

} // namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser;

    auto mrcConfigPath = parser.string("mrc-config")
            .help("path to mrc config");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto featureIdsFileName = parser.string("input")
        .required()
        .help("Input file path");

    auto outputFileName = parser.string("output")
        .required()
        .help("Output file path");

    parser.parse(argc, const_cast<char**>(argv));

    const auto config = mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);
    auto poolHolder = config.makePoolHolder();

    auto features = loadFeatures(poolHolder.pool(), loadFeatureIds(featureIdsFileName));

    maps::http::Client httpClient;

    std::ofstream ofs(outputFileName);
    json::Builder builder(ofs);
    builder << [&](json::ObjectBuilder b) {
        b[JSON_NAME_IMAGES_SET] = [&](json::ArrayBuilder b) {
            for (const auto& feature : features) {
                b << [&](json::ObjectBuilder b) {
                    b[JSON_NAME_FEATURE_ID] = feature.id();
                    b[JSON_NAME_LAT] = feature.geodeticPos().y();
                    b[JSON_NAME_LON] = feature.geodeticPos().x();
                    b[JSON_NAME_IMAGE] = encodeImage(loadFeatureImage(httpClient, feature));
                    b[JSON_NAME_OBJECTS] = [&](json::ArrayBuilder) {};
                };
            }
        };
    };

    ofs.close();
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
