#include <maps/wikimap/mapspro/services/mrc/libs/privacy_detector/include/privacy_detector_faster_rcnn.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/http/include/http.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/value.h>

#include <opencv2/opencv.hpp>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <chrono>

using namespace std::chrono;
using namespace maps::mrc::privacy_detector;

namespace {

struct Feature {
    int64_t featureID;
    std::string url;
};


std::set<int64_t> loadFilterFeatureIDs(const std::string &filterPath) {
    std::set<int64_t> filterFeatureIDs;
    std::ifstream ifs(filterPath);
    if (!ifs.is_open())
        return filterFeatureIDs;
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        filterFeatureIDs.insert(std::stoll(line));
    }
    return filterFeatureIDs;
}

std::string download(maps::http::Client& client, const std::string& url, size_t retryCnt) {
    maps::common::RetryPolicy retryPolicy;
    retryPolicy.setTryNumber(retryCnt);

    auto validateResponse = [](const auto& maybeResponse) {
        return maybeResponse.valid() && maybeResponse.get().responseClass() != maps::http::ResponseClass::ServerError;
    };
    auto resp = maps::common::retry(
                [&]() {
                    return maps::http::Request(client, maps::http::GET, maps::http::URL(url)).perform();
                },
                retryPolicy,
                validateResponse
            );
    if (resp.responseClass() != maps::http::ResponseClass::Success) {
        INFO() << "Unexpected response status " << resp.status() << " for url " << url;
        return "";
    }
    return resp.readBody();
}


void saveFeatureWithObjects(const Feature &feature, const PrivacyImageBoxes &objects, maps::json::Builder &builder) {
    builder << [&feature, &objects](maps::json::ObjectBuilder builder) {
        builder["feature_id"] << feature.featureID;
        builder["source"]     << feature.url;
        builder["facesPresent"] << std::any_of(objects.begin(), objects.end(), [](PrivacyImageBox box){return box.type == PrivacyImageBox::Type::Face;});
        builder["platesPresent"] << std::any_of(objects.begin(), objects.end(), [](PrivacyImageBox box){return box.type == PrivacyImageBox::Type::LicensePlate;});
    };
}


} //namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Launch detection of private data (faces and license plates) on images");

    maps::cmdline::Option<std::string> inputPath = parser.string("input")
        .required()
        .help("Path to list of images in text file with one image on the line:\
               <feature_id> <mds_url>");

    maps::cmdline::Option<std::string> outputPath = parser.string("output")
        .required()
        .help("Path to file for save results. Results for one image on the line in json format");

    maps::cmdline::Option<std::string> filterPath = parser.string("filter")
        .defaultValue("")
        .help("Path to text file with feature ids witch already checked");

    maps::cmdline::Option<size_t> fromIndex = parser.size_t("from_idx")
        .defaultValue(0)
        .help("Index of start line in input file");

    maps::cmdline::Option<size_t> toIndex = parser.size_t("to_idx")
        .defaultValue(-1)
        .help("Index of line after last processed in input file");

    maps::cmdline::Option<size_t> retryCnt = parser.size_t("retry_cnt")
        .defaultValue(20)
        .help("Retry for download image");

    parser.parse(argc, const_cast<char**>(argv));

    std::set<int64_t> filterFeatureIDs = loadFilterFeatureIDs(filterPath);

    std::ifstream ifs(inputPath);
    std::string line;
    size_t inpLineIdx = 0;
    for (; inpLineIdx < fromIndex && !ifs.eof(); inpLineIdx++) {
        std::getline(ifs, line);
    }

    INFO() << inpLineIdx << " skipped";

    maps::http::Client client;
    std::ofstream reworked(filterPath, std::ofstream::out | std::ofstream::app);
    std::ofstream ofs(outputPath, std::ofstream::out | std::ofstream::app);
    Feature feature;
    FasterRCNNDetector detector;
    for (; inpLineIdx < toIndex && !ifs.eof(); inpLineIdx++) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        std::stringstream ss(line);
        ss >> feature.featureID
           >> feature.url;
        if (filterFeatureIDs.find(feature.featureID) != filterFeatureIDs.end()) {
            INFO() << feature.featureID << " skipped (as already processed)";
            continue;
        }
        std::string temp = download(client, feature.url, retryCnt);
        if (temp.empty()) {
            INFO() << feature.featureID << " skipped (no data downloaded by url)";
            continue;
        }
        cv::Mat encImage((int)temp.size(), 1, CV_8UC1, (void *)temp.c_str());
        cv::Mat image;
        try {
            image = cv::imdecode(encImage, cv::IMREAD_COLOR);
        } catch (...){
            INFO() << feature.featureID << " skipped (unable to decode image from url: " << feature.url << ")";
            continue;
        }
        if (image.empty()) {
            INFO() << feature.featureID << " skipped (unable to decode image from url: " << feature.url << ")";
            continue;
        }

        PrivacyImageBoxes privacyObjects = detector.detect(image);
        std::stringstream jsonLine;
        maps::json::Builder builder(jsonLine);
        saveFeatureWithObjects(feature, privacyObjects, builder);
        ofs << jsonLine.str() << std::endl; ofs.flush();
        reworked << feature.featureID << std::endl; reworked.flush();
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
