#include "tool.h"

#include <contrib/libs/geos/include/geos/geom/Polygon.h>
#include <contrib/libs/geos/include/geos/io/WKBReader.h>
#include <mapreduce/yt/interface/client.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/parallel_for_each.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <boost/filesystem.hpp>

#include <fstream>
#include <sstream>
#include <string_view>

namespace maps::mrc {
namespace {

const std::string IMAGE_SUBDIR = "images";

geolib3::Polygon2 hexWKBToPolygon(std::string_view hexWKB)
{
    std::stringstream ss(std::string{hexWKB});
    std::unique_ptr<geos::geom::Geometry> geom(
        geos::io::WKBReader().readHEX(ss));
    REQUIRE(dynamic_cast<geos::geom::Polygon*>(geom.get()),
            "Incorrect geometry type");
    return geolib3::internal::geos2geolibGeometry(
        dynamic_cast<geos::geom::Polygon*>(geom.get()));
}

geolib3::Polygon2 loadGeodeticAoi(const std::string& ytTable, long long adId)
{
    INFO() << "Connecting to yt";
    NYT::IClientPtr client = NYT::CreateClient("hahn");
    NYT::ITransactionPtr txn = client->StartTransaction();
    std::string path =
        "//home/maps/core/garden/stable/ymapsdf/latest/" + ytTable + "/ad_geom";
    NYT::TRichYPath range(path.c_str());
    range.AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(adId)));
    NYT::TTableReaderPtr<NYT::TNode> reader =
        txn->CreateTableReader<NYT::TNode>(range);
    for (; reader->IsValid(); reader->Next()) {
        auto result = hexWKBToPolygon(reader->GetRow()["shape"].AsString());
        auto bbox = result.boundingBox();
        INFO() << "BBOX(" << bbox.minX() << " " << bbox.minY() << ", "
               << bbox.maxX() << " " << bbox.maxY() << ")";
        return result;
    }
    throw RuntimeError("Failed to load AOI");
}

auto loadFeatures(sql_chemistry::Transaction& txn,
                  const geolib3::Polygon2& mercatorGeom)
{
    INFO() << "loading database features";
    auto result = db::FeatureGateway(txn).load(
        db::table::Feature::isPublished &&
        db::table::Feature::pos.intersects(mercatorGeom.boundingBox()));
    result.erase(
        std::remove_if(
            result.begin(),
            result.end(),
            [&](const auto& feature) {
                return feature.privacy() == db::FeaturePrivacy::Secret ||
                       !spatialRelation(mercatorGeom,
                                        feature.mercatorPos(),
                                        geolib3::SpatialRelation::Intersects);
            }),
        result.end());
    std::sort(
        result.begin(), result.end(), [](const auto& lhs, const auto& rhs) {
            return std::make_tuple(lhs.sourceId(), lhs.timestamp(), lhs.id()) <
                   std::make_tuple(rhs.sourceId(), rhs.timestamp(), rhs.id());
        });
    INFO() << "loaded " << result.size() << " database features";
    return result;
}

struct FeatureManipulator {
    const db::Feature& feature;

    std::string jpg() const
    {
        std::ostringstream result;
        result << feature.id() << ".jpg";
        return result.str();
    }

    std::string url() const
    {
        std::ostringstream result;
        result << "https://core-nmaps-mrc-browser.maps.yandex.ru/feature/"
               << feature.id() << "/image";
        return result.str();
    }

    void save(const std::string& outDir,
              const std::string& sessionId,
              const std::string& sessionId2) const
    {
        constexpr int RETRY_NUMBER(10);
        constexpr std::chrono::seconds RETRY_TIMEOUT(3);

        std::string filePath = outDir + "/" + IMAGE_SUBDIR + "/" + jpg();
        if (boost::filesystem::exists(filePath)) {
            return;
        }

        http::Client client;
        http::Request request{client, http::GET, url()};
        request.addHeader(
            "COOKIE",
            "Session_id=" + sessionId + "; sessionid2=" + sessionId2 + ";");
        for (int i = 0; i < RETRY_NUMBER; i++) {
            if (0 != i)
                std::this_thread::sleep_for(RETRY_TIMEOUT);
            try {
                http::Response response = request.perform();
                if (200 == response.status()) {
                    maps::common::writeFile(filePath, response.readBody());
                    return;
                }
                WARN() << response.status();
            }
            catch (const Exception& e) {
                WARN() << e;
            }
        }
        throw RuntimeError("Failed to download image ") << url();
    }

    void json(json::ObjectBuilder b) const
    {
        b["type"] = "Feature";
        b["id"] = feature.id();
        b["geometry"] << geolib3::geojson(feature.geodeticPos());
        b["properties"] = [&](json::ObjectBuilder b) {
            b["image"] = IMAGE_SUBDIR + "/" + jpg();
            b["heading"] = (int)feature.heading().value();
            b["source_id"] = feature.sourceId();
            b["date"] = chrono::formatIsoDateTime(feature.timestamp());
        };
    }
};

void save(const std::string& outDir,
          const std::string& sessionId,
          const std::string& sessionId2,
          const db::Features& features)
{
    INFO() << "loading MDS features";
    int counter = 0;
    common::parallelForEach(features.begin(), features.end(),
        [&](auto& guard, const auto& feature) {
            FeatureManipulator{feature}.save(outDir, sessionId, sessionId2);
            std::lock_guard lock{guard};
            ++counter;
            if (counter % 1000 == 0) {
                INFO() << "loaded " << counter << " MDS features";
            }
        });
    INFO() << "loaded " << counter << " MDS features";
}

void toJson(const std::string& outDir, const db::Features& features)
{
    std::ofstream os(outDir + "/result.json");
    json::Builder builder(os);
    builder << [&](json::ObjectBuilder b) {
        b["data"] << [&](json::ObjectBuilder b) {
            b["type"] = "FeatureCollection";
            b["features"] << [&](json::ArrayBuilder b) {
                for (const auto& feature : features) {
                    b << FeatureManipulator{feature};
                }
            };
        };
    };
}

}  // namespace

void run(common::Config& cfg,
         const std::string& ytTable,
         long long adId,
         const std::string& outDir,
         const std::string& sessionId,
         const std::string& sessionId2)
{
    std::string subDir = outDir + "/" + ytTable + "_" + std::to_string(adId);
    boost::filesystem::create_directory(subDir);
    boost::filesystem::create_directory(subDir + "/" + IMAGE_SUBDIR);
    auto mercatorGeom =
        convertGeodeticToMercator(loadGeodeticAoi(ytTable, adId));
    auto features = loadFeatures(
        *cfg.makePoolHolder().pool().slaveTransaction(), mercatorGeom);
    save(subDir, sessionId, sessionId2, features);
    toJson(subDir, features);
}

}  // namespace maps::mrc
