#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/include/types.h>
#include <maps/wikimap/mapspro/services/mrc/libs/sensors_feature_positioner/include/types.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/image_box.h>

#include <maps/libs/tagged/include/tagged.h>
#include <maps/libs/geolib/include/direction.h>

#include <opencv2/opencv.hpp>

#include <optional>
#include <list>

namespace maps::mrc::experiment_sfm_positioning {

namespace unit {
    struct MercatorUnits;
    struct Meters;
} // namespace units
using MercatorUnits =  tagged::Tagged<unit::MercatorUnits, double>;
using Meters =  tagged::Tagged<unit::Meters, double>;

struct ObjectInfo {
    Meters radius;
    common::ImageBox bbox;
};
using ObjectsInfo = std::list<ObjectInfo>;

struct PositionedObject {
    const ObjectInfo& object1;
    const ObjectInfo& object2;

    geolib3::Point2 mercatorPos;
};
using PositionedObjects = std::vector<PositionedObject>;

struct CameraPose {
    // Camera position in Mercator projection
    geolib3::Point2 mercatorPos;
    // 3x3 camera rotation matrix to translate to global coordinate system
    // with the basis vectors: X - East, Y - Down, Z - North. Its elements are
    // expected to be of CV_32F type.
    // See https://docs.opencv.org/3.2.0/pinhole_camera_model.png
    cv::Mat rotation;
};

// Compute camera and objects position. It is expected that the input images
// are rectified and the both are taken with the same camera.
// https://st.yandex-team.ru/MAPSMRC-511
//
// Employ BRISK features as was decided by earlier evaluation with the
// inferred limitations. Images are taken in {3, 30} range (in meters) from
// each other and positioned objects aren't further then 40 meters away from
// camera positions. Also filter out objects that positioned behind a camera
// (with negative Z value).
//
// Returns a list of positioned objects relative to the first camera position
//
// See the following comment
// https://st.yandex-team.ru/MAPSMRC-511#5a7c678c141a11001a0f1092

PositionedObjects computeObjectPositionsWith2Images(
    cv::Mat camMat,                // a camera matrix (same for the both images)
    cv::Mat img1,                  // the 1st image
    cv::Mat img1Mask,              // the 1st image features mask
    const CameraPose& cam1GeoPose, // the 1st camera position
    const ObjectsInfo& objects1,   // the 1st image objects
    cv::Mat img2,                  // the 2nd image
    cv::Mat img2Mask,              // the 2nd image features mask
    const CameraPose& cam2GeoPose, // the 2nd camera position
    const ObjectsInfo& objects2,   // the 2nd image objects
    std::size_t featuresLimit);

// Check the camera move and turn is within allowed range before to call to
// the heavy computeObjectPositionsWith2Images.
bool isCameraMoveInAllowedRange(
    const CameraPose& pose1, const CameraPose& pose2);

inline Meters toMeters(const geolib3::Point2& pos, MercatorUnits mercatorUnits)
{
    auto mercatorRatio = geolib3::MercatorRatio::fromMercatorPoint(pos);
    return Meters{mercatorRatio.toMeters(mercatorUnits.value())};
}

} // namespace maps::mrc::experiment_sfm_positioning
