#include <maps/wikimap/mapspro/services/mrc/tools/experiment_sign_position_accuracy/lib/include/accuracy.h>

#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>

#include <unordered_set>

namespace maps::mrc::tracks_with_sensors {

Accuracy compareWithGroundTruth(const Signs& signs,
                                const Signs& gtSigns,
                                SignMatching signMatching)
{
    std::unordered_set<db::TId> foundGtSigns;
    int severalSignsForOneGtCounter = 0;
    double distancesSum = 0;
    std::vector<double> errors;
    for (const auto& sign : signs) {
        if (signMatching == SignMatching::ById) {
            REQUIRE(sign.id, "sign should have id");
        }
        double bestDistance = std::numeric_limits<double>::max();
        std::optional<Sign> bestGtSign;
        for (size_t i = 0; i < gtSigns.size(); i++) {
            if (gtSigns[i].signType != sign.signType) {
                continue;
            }
            double curDistance = geolib3::toMeters(
                geolib3::distance(gtSigns[i].mercatorPos, sign.mercatorPos),
                sign.mercatorPos);
            if (signMatching == SignMatching::Nearest) {
                if (curDistance < bestDistance) {
                    bestGtSign = gtSigns[i];
                    bestDistance = curDistance;
                }
            } else {
                REQUIRE(gtSigns[i].id, "gtSign should have id");
                if (*gtSigns[i].id == *sign.id) {
                    bestGtSign = gtSigns[i];
                    bestDistance = curDistance;
                }
            }
        }

        if (!bestGtSign && signMatching == SignMatching::ById) {
            ERROR() << "Didn't find GT sign with ID " << *sign.id;
            continue;
        } else {
            REQUIRE(bestGtSign, "can't find gt sign for the provided sign");
        }

        if (foundGtSigns.count(*bestGtSign->id)) {
            severalSignsForOneGtCounter++;
        }
        foundGtSigns.insert(*bestGtSign->id);
        distancesSum += bestDistance;
        errors.push_back(bestDistance);
    }
    Accuracy resultAccuracy;
    resultAccuracy.avgError = distancesSum / errors.size();
    resultAccuracy.recall = foundGtSigns.size() / (double(gtSigns.size()));
    std::sort(errors.begin(), errors.end());
    resultAccuracy.sortedErrors = errors;
    for (double error : errors) {
        resultAccuracy.deviation +=
            (error - resultAccuracy.avgError) * (error - resultAccuracy.avgError);
    }
    resultAccuracy.deviation /= errors.size();
    resultAccuracy.deviation = std::sqrt(resultAccuracy.deviation);
    resultAccuracy.severalSignsForOneGt = severalSignsForOneGtCounter;
    return resultAccuracy;
}

void printAccuracy(const Accuracy& accuracy) {
    INFO() << "avg accuracy = " << accuracy.avgError;
    INFO() << "deviation = " << accuracy.deviation;
    INFO() << "recall = " << accuracy.recall;
    INFO() << "several signs matched to one gt sign: " << accuracy.severalSignsForOneGt;

    if (!accuracy.sortedErrors.empty()) {
        INFO() << "accuracy quantiles:";
        for (int i = 0; i <= 100; i += 5) {
            INFO() << i << "%: "
                   << accuracy.sortedErrors[(accuracy.sortedErrors.size() - 1)
                          * i / 100];
        }
    }
}

} // namespace maps::mrc::tracks_with_sensors
