#include <maps/wikimap/mapspro/services/mrc/tools/experiment_sign_position_accuracy/lib/include/db_loader.h>

#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/include/position_improvment.h>
#include <maps/libs/log8/include/log8.h>

namespace maps::mrc::tracks_with_sensors {

db::TrackPoints loadTrack(
    wiki::common::PoolHolder& poolHolder,
    db::TId assignmentId,
    std::string sourceId)
{
    auto txn = poolHolder.pool().slaveTransaction();
    db::TrackPointGateway tpGtw(*txn);
    db::TrackPoints trackPoints = tpGtw.load(
        db::table::TrackPoint::assignmentId.equals(assignmentId)
        && db::table::TrackPoint::sourceId.equals(sourceId));

    if (trackPoints.empty()) {
        return {};
    }

    INFO() << "loaded " << trackPoints.size() << "trackPoints";
    std::sort(trackPoints.begin(), trackPoints.end(),
              [](const db::TrackPoint& lhs, const db::TrackPoint& rhs) {
                  return lhs.timestamp() < rhs.timestamp();
              });

    return trackPoints;
}

Photos loadPhotos(
    wiki::common::PoolHolder& poolHolder,
    db::TId assignmentId,
    std::string sourceId,
    chrono::TimePoint minTime,
    chrono::TimePoint maxTime)
{
    auto txn = poolHolder.pool().slaveTransaction();
    db::FeatureGateway photoGtw(*txn);

    db::Features photos = photoGtw.load(
        db::table::Feature::assignmentId.equals(assignmentId)
        && db::table::Feature::sourceId.equals(sourceId)
        && db::table::Feature::date >= minTime
        && db::table::Feature::date <= maxTime);
    INFO() << "loaded " << photos.size() << "assignmentPhotos";

    std::sort(photos.begin(), photos.end(),
              [](const db::Feature& lhs, const db::Feature& rhs) {
                  return lhs.timestamp() < rhs.timestamp();
              });

    Photos filteredPhotos;
    filteredPhotos.reserve(photos.size());

    for (auto photo : photos) {
        if (!filteredPhotos.empty() &&
            photo.timestamp() == filteredPhotos.back().timestamp)
        {
            WARN() << "2 photos with equal timestamp";
            continue;
        }

        if (!filteredPhotos.empty() &&
            photo.timestamp() - filteredPhotos.back().timestamp < std::chrono::milliseconds(10))
        {
            WARN() << "2 photos with very similar timestamp";
            continue;
        }

        filteredPhotos.push_back(Photo{
                photo.id(),
                photo.timestamp(),
                photo.mercatorPos(),
                photo.heading(),
                photo.size(),
                photo.orientation()});
    }

    INFO() << "loaded " << filteredPhotos.size() << "filteredPhotos";
    return filteredPhotos;
}

SignPhotos loadSignBboxes(
    wiki::common::PoolHolder& poolHolder,
    const Photos& photos)
{
    auto txn = poolHolder.pool().slaveTransaction();
    db::SignFeatureGateway signFeatureGtw(*txn);

    std::vector<db::TId> featureIds;
    std::map<db::TId, const Photo*> featuresMap;
    for (auto& photo : photos) {
        featureIds.push_back(photo.featureId);
        featuresMap[photo.featureId] = &photo;
    }
    db::SignFeatures signFeatures = signFeatureGtw.load(
        db::table::SignFeature::featureId.in({featureIds}));

    std::vector<db::TId> signIds;
    for (const auto& signFeature : signFeatures) {
        signIds.push_back(signFeature.signId());
    }
    db::SignGateway signGtw(*txn);
    db::Signs signs = signGtw.load(db::table::Sign::id.in({signIds}));

    std::map<db::TId, db::Sign*> signsMap;
    for (auto& sign : signs) {
        signsMap[sign.id()] = &sign;
    }

    SignPhotos signPhotos;
    signPhotos.reserve(signFeatures.size());
    for (const auto& signFeature : signFeatures) {
        auto imageBox = revertByImageOrientation(
            signFeature.imageBox(),
            featuresMap[signFeature.featureId()]->size,
            featuresMap[signFeature.featureId()]->orientation);

        signPhotos.push_back(SignPhoto{
                signFeature.featureId(),
                featuresMap[signFeature.featureId()]->timestamp,
                featuresMap[signFeature.featureId()]->size,
                signsMap[signFeature.signId()]->type(),
                imageBox,
                std::nullopt});
    }

    INFO() << "loaded " << signPhotos.size() << "signs Bboxes";
    return signPhotos;
}

db::Signs loadSigns(
    wiki::common::PoolHolder& poolHolder,
    const geolib3::BoundingBox& mercatorBbox)
{
    auto txn = poolHolder.pool().slaveTransaction();
    db::SignGateway signGtw(*txn);

    return signGtw.load(db::table::Sign::position.intersects(mercatorBbox));
}

db::Features loadFeaturesOfSign(
    wiki::common::PoolHolder& poolHolder,
    db::TId signId)
{
    auto txn = poolHolder.pool().slaveTransaction();
    db::SignFeatureGateway signFeatureGtw(*txn);

    db::SignFeatures signFeatures = signFeatureGtw.load(
        db::table::SignFeature::signId.equals(signId));

    std::vector<db::TId> featureIds;
    for (const auto& signFeature : signFeatures) {
        featureIds.push_back(signFeature.featureId());
    }
    db::FeatureGateway featureGtw(*txn);
    db::Features features = featureGtw.load(db::table::Feature::id.in({featureIds}));

    return features;
}


} // namespace maps::mrc::tracks_with_sensors
