#include <maps/wikimap/mapspro/services/mrc/tools/experiment_sign_position_accuracy/lib/include/geojson.h>

#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/vector.h>
#include <maps/libs/log8/include/log8.h>

#define _USE_MATH_DEFINES

#include <fstream>
#include <string>
#include <vector>
#include <cmath>
#include <unordered_set>

namespace maps::mrc::tracks_with_sensors {

namespace sfp = maps::mrc::sensors_feature_positioner;

void GeojsonSaver::addSignGroupAsPolyline(const Signs& signGroup, Properties properties)
{
    std::vector<geolib3::Point2> points;
    for(const auto& sign : signGroup) {
        points.push_back(mercator2GeoPoint(sign.mercatorPos));
    }

    points.push_back(geolib3::fastGeoShift(
                         mercator2GeoPoint(signGroup[0].mercatorPos),
                         geolib3::Vector2(0.1, 0)));

    polylines_.push_back({properties, geolib3::Polyline2(points)});
}

void GeojsonSaver::addSignGroupsAsPolylines(const std::vector<Signs>& signGroups)
{
    std::vector<std::string> colors = {"#0000ff"};

    for (size_t i = 0; i < signGroups.size(); i++) {
        std::string tag = std::to_string(signGroups[i].size()) + " "
            + traffic_signs::toString(signGroups[i][0].signType);
        Properties properties {tag,
                               colors[i % colors.size()],
                               5.0};
        addSignGroupAsPolyline(signGroups[i], properties);
    }
}

void GeojsonSaver::addSignAsRay(const sfp::Ray& signRay)
{

    Properties properties {traffic_signs::toString(static_cast<traffic_signs::TrafficSign>(signRay.objectTypeId)),
                           "#ff0000",
                           1.0};
    std::vector<geolib3::Point2> points;
    double dy = 80 * signRay.directionToObject.y();
    double dx = 80 * signRay.directionToObject.x();
    points.push_back(mercator2GeoPoint(signRay.cameraPos.mercatorPosition()));
    points.push_back(fastGeoShift(mercator2GeoPoint(signRay.cameraPos.mercatorPosition()),
                                  geolib3::Vector2(dx, dy)));
    polylines_.push_back({properties, geolib3::Polyline2(points)});

    properties.color = "#0000ff";
    points = {};
    dy = signRay.metersToObject * signRay.directionToObject.y();
    dx = signRay.metersToObject * signRay.directionToObject.x();
    points.push_back(mercator2GeoPoint(signRay.cameraPos.mercatorPosition()));
    points.push_back(fastGeoShift(mercator2GeoPoint(signRay.cameraPos.mercatorPosition()),
                                  geolib3::Vector2(dx, dy)));
    polylines_.push_back({properties, geolib3::Polyline2(points)});
}

void GeojsonSaver::addPhotoAsRay(const pos_improvment::ImprovedGpsEvent& pos)
{

    Properties properties {"photo",
                           "#ff0000",
                           1.0};
    std::vector<geolib3::Point2> points;
    double dy = 10 * pos.cameraFrontDirection().y();
    double dx = 10 * pos.cameraFrontDirection().x();
    points.push_back(mercator2GeoPoint(pos.mercatorPosition()));
    points.push_back(fastGeoShift(mercator2GeoPoint(pos.mercatorPosition()),
                                  geolib3::Vector2(dx, dy)));
    polylines_.push_back({properties, geolib3::Polyline2(points)});
}

void GeojsonSaver::addSignsAsRays(const sfp::Rays& signRays)
{
    std::unordered_set<int> addedDbSigns;
    for (const auto& signRay : signRays) {
        addSignAsRay(signRay);
    }
}

void GeojsonSaver::addSignsAsPoints(const Signs& signs,
                                    const std::string& tag,
                                    const std::string& color)
{
    for (const auto& sign : signs) {
        std::string signId = sign.id ? std::to_string(*sign.id) : "no id";

        Properties properties {tag + " " + signId + " " + traffic_signs::toString(sign.signType),
                color,
                5.0};
        std::vector<geolib3::Point2> points;
        points.push_back(mercator2GeoPoint(sign.mercatorPos));
        points.push_back(geolib3::fastGeoShift(
                             mercator2GeoPoint(sign.mercatorPos),
                         geolib3::Vector2(0.1, 0)));
        polylines_.push_back({properties, geolib3::Polyline2(points)});
    }
}

std::string createGeojsonPolyline(size_t& id,
                                  const geolib3::Polyline2& polyline,
                                  Properties properties)
{
    json::Builder builder;
    builder << [&](json::ObjectBuilder b) {
        b["type"] = "Feature";
        b["id"] = id;
        b["geometry"] << geolib3::geojson(polyline);
        b["properties"] = [&](json::ObjectBuilder b) {
            b["description"] = properties.description;
            b["stroke"] = properties.color;
            b["stroke-width"] = properties.width;
            b["stroke-opacity"] = 1.0;
        };
    };
    id++;
    return builder.str();
}

void GeojsonSaver::save(const std::string& filename)
{
    size_t id = 0;
    json::Builder builder;
    builder << [&](json::ObjectBuilder b) {
        b["type"] = "FeatureCollection";
        b["features"] << [&](json::ArrayBuilder b) {
            for (const auto& it : polylines_) {
                b << json::Verbatim(createGeojsonPolyline(
                                        id, it.second, it.first));
            }
        };
    };
    std::ofstream outFile(filename);
    outFile << builder.str();
}

} // namespace maps::mrc::tracks_with_sensors
