#include <maps/wikimap/mapspro/services/mrc/tools/experiment_sign_position_accuracy/lib/include/outdated_signs.h>

#include <maps/wikimap/mapspro/services/mrc/tools/experiment_sign_position_accuracy/lib/include/db_loader.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/visibility.h>

#include <boost/geometry.hpp>
#include <boost/geometry/geometries/point_xy.hpp>
#include <boost/geometry/index/rtree.hpp>

namespace maps::mrc::tracks_with_sensors {

using BPoint = boost::geometry::model::d2::point_xy<double>;
using BBox = boost::geometry::model::box<BPoint>;

struct IndexableFeature {
    using result_type = BPoint;

    BPoint operator()(const db::Feature& photo) const
    {
        return {photo.mercatorPos().x(), photo.mercatorPos().y()};
    }
};

struct IndexableSign {
    using result_type = BPoint;

    BPoint operator()(const Sign& sign) const
    {
        return {sign.mercatorPos.x(), sign.mercatorPos.y()};
    }
};

struct IndexableDbSign {
    using result_type = BPoint;

    BPoint operator()(const db::Sign& sign) const
    {
        return {sign.mercatorPos().x(), sign.mercatorPos().y()};
    }
};


using FeatureRTree
    = boost::geometry::index::rtree<db::Feature,
                                    boost::geometry::index::quadratic<16>,
                                    IndexableFeature>;
using SignRTree
    = boost::geometry::index::rtree<Sign,
                                    boost::geometry::index::quadratic<16>,
                                    IndexableSign>;
using DbSignRTree
= boost::geometry::index::rtree<db::Sign,
                                    boost::geometry::index::quadratic<16>,
                                    IndexableDbSign>;

BPoint toBoost(const geolib3::Point2& point)
{
    return BPoint(point.x(), point.y());
}

BBox toBoost(const geolib3::BoundingBox& bbox)
{
    return BBox{toBoost(bbox.lowerCorner()), toBoost(bbox.upperCorner())};
}


geolib3::BoundingBox expandMercatorBbox(geolib3::BoundingBox bbox,
                                        double extensionMeters)
{
    geolib3::Point2 leftBottom = bbox.lowerCorner();
    geolib3::Point2 rightTop = bbox.upperCorner();

    geolib3::Point2 leftBottomGeo = geolib3::mercator2GeoPoint(leftBottom);
    geolib3::Point2 rightTopGeo = geolib3::mercator2GeoPoint(rightTop);

    leftBottomGeo = geolib3::fastGeoShift(
        leftBottomGeo, geolib3::Vector2(-extensionMeters, -extensionMeters));
    rightTopGeo = geolib3::fastGeoShift(
        rightTopGeo, geolib3::Vector2(extensionMeters, extensionMeters));

    leftBottom = geolib3::geoPoint2Mercator(leftBottomGeo);
    rightTop = geolib3::geoPoint2Mercator(rightTopGeo);
    return geolib3::BoundingBox(leftBottom, rightTop);
}

geolib3::BoundingBox featuresMercatorBbox(const db::Features& photos,
                                          double extensionMeters)
{
    REQUIRE(photos.size(), "not photos provided");
    double minX = photos.front().mercatorPos().x();
    double maxX = photos.front().mercatorPos().x();
    double minY = photos.front().mercatorPos().y();
    double maxY = photos.front().mercatorPos().y();

    for (const auto& photo : photos) {
        minX = std::min(minX, photo.mercatorPos().x());
        maxX = std::max(maxX, photo.mercatorPos().x());
        minY = std::min(minY, photo.mercatorPos().y());
        maxY = std::max(maxY, photo.mercatorPos().y());
    }

    return expandMercatorBbox(
        geolib3::BoundingBox(geolib3::Point2(minX, minY),
                             geolib3::Point2(maxX, maxY)),
        extensionMeters);
}

geolib3::BoundingBox getVisibilityArea(const db::Sign& sign)
{
    return expandMercatorBbox(geolib3::BoundingBox(sign.mercatorPos(), 0, 0),
                              100);
}

// checks if there is a similar sign with 90 degress different heading
bool perpendicularSignExists(const db::Sign& sign,
                             const db::Signs& candidates)
{
    for (const db::Sign& candidate : candidates) {
        if (candidate.type() == sign.type()
            && geolib3::fastGeoDistance(
                geolib3::mercator2GeoPoint(sign.mercatorPos()),
                geolib3::mercator2GeoPoint(candidate.mercatorPos())) < 50
            && (geolib3::angleBetween(geolib3::Direction2(sign.heading()),
                                      geolib3::Direction2(candidate.heading()))
                > geolib3::PI / 3)
            && (geolib3::angleBetween(geolib3::Direction2(sign.heading()),
                                      geolib3::Direction2(candidate.heading()))
                < geolib3::PI / 3 * 4))
        {
            return true;
        }
    }
    return false;
}

// check there is a similar old sign with smaller id
bool similarSignExists(const db::Sign& sign,
                       const db::Signs& candidates)
{
    for (const db::Sign& candidate : candidates) {
        if (candidate.type() == sign.type()
            && geolib3::fastGeoDistance(
                geolib3::mercator2GeoPoint(sign.mercatorPos()),
                geolib3::mercator2GeoPoint(candidate.mercatorPos())) < 50
            && (geolib3::angleBetween(geolib3::Direction2(sign.heading()),
                                      geolib3::Direction2(candidate.heading()))
                < geolib3::PI / 4)
            && (sign.id() < candidate.id()))
        {
            return true;
        }
    }
    return false;
}



// check if newSigns contain similar sign
bool signStillExists(const db::Sign& sign,
                     const Signs& newSigns)
{
    for (const Sign& candidate : newSigns) {
        if (candidate.signType == sign.type()
            && geolib3::fastGeoDistance(
                geolib3::mercator2GeoPoint(sign.mercatorPos()),
                geolib3::mercator2GeoPoint(candidate.mercatorPos)) < 60
            && (geolib3::angleBetween(geolib3::Direction2(sign.heading()),
                                      geolib3::Direction2(candidate.heading))
                < geolib3::PI / 2))
        {
            return true;
        }
    }
    return false;
}

bool signIsVisible(const db::Sign& sign,
                   const db::Feature& feature) {
    geolib3::Vector2 signDirection(
        geolib3::cos(geolib3::Direction2(sign.heading()).radians()),
        geolib3::sin(geolib3::Direction2(sign.heading()).radians()));

    geolib3::Segment2 signAsSegment(
        geolib3::mercator2GeoPoint(sign.mercatorPos()),
        geolib3::fastGeoShift(geolib3::mercator2GeoPoint(sign.mercatorPos()),
                              signDirection * -0.1));

    db::Segments segments{signAsSegment};
    db::Segments result = db::getUncovered(feature, segments);
    return result.empty();
}

bool mostPhotosAreSideDirected(const db::Features& features) {
    size_t sidePhotos = 0;
    for (const db::Feature& feature : features) {
        if (feature.hasCameraDeviation()
            && feature.cameraDeviation() != db::CameraDeviation::Front)
        {
            sidePhotos++;
        }
    }
    if (sidePhotos >= features.size()) {
        return true;
    } else {
        return false;
    }
}

bool featureAfterDateExist(const db::Features& features,
                           chrono::TimePoint date) {
    for (const db::Feature& feature : features) {
        if (feature.timestamp() > date) {
            return true;
        }
    }
    return false;
}


db::Features getPhotosWhereSignIsVisible(const db::Sign& sign,
                                         const db::Features& photos)
{
    db::Features adjacentPhotos;
    for (const db::Feature& photo : photos) {
        if (signIsVisible(sign, photo)
            && (geolib3::angleBetween(geolib3::Direction2(sign.heading()),
                                      geolib3::Direction2(geolib3::reverse(photo.heading())))
                < geolib3::PI / 3))
        {
            adjacentPhotos.push_back(photo);
        }
    }
    return adjacentPhotos;
}

db::Signs findOutdatedSigns(wiki::common::PoolHolder& poolHolder,
                            const db::Features& newPhotos,
                            const Signs& newSigns)
{
    db::Signs outdatedSigns;

    auto newTrackBbox = featuresMercatorBbox(newPhotos, 100);
    db::Signs oldSigns = loadSigns(poolHolder, newTrackBbox);

    FeatureRTree newPhotosRTree{newPhotos.begin(), newPhotos.end()};
    SignRTree newSignsRTree{newSigns.begin(), newSigns.end()};
    DbSignRTree oldSignsRTree{oldSigns.begin(), oldSigns.end()};

    for (const db::Sign oldSign : oldSigns) {
        if (traffic_signs::isRoadMarking(oldSign.type())) {
            continue;
        }
        geolib3::BoundingBox searchingArea = getVisibilityArea(oldSign);

        db::Signs oldNeighboringSigns;
        oldSignsRTree.query(
            boost::geometry::index::intersects(toBoost(searchingArea)),
            std::back_inserter(oldNeighboringSigns));
        if (perpendicularSignExists(oldSign, oldNeighboringSigns)) {
            // there is the same sign with a different heading. Maybe
            // the current oldSign is on crossroad and has wrong
            // heading because it was captured from perpendicular road.
            // It is better to skip such signs;
            continue;
        }

        if (similarSignExists(oldSign, oldNeighboringSigns)) {
            // for some reason there are several similar old signs.
            // Lets check only the one with the smallest id
            continue;
        }

        Signs newNeighboringSigns;
        newSignsRTree.query(
            boost::geometry::index::intersects(toBoost(searchingArea)),
            std::back_inserter(newNeighboringSigns));
        if (signStillExists(oldSign, newNeighboringSigns)) {
            continue;
        }

        db::Features neighboringPhotos;
        newPhotosRTree.query(
            boost::geometry::index::intersects(toBoost(searchingArea)),
            std::back_inserter(neighboringPhotos));
        db::Features photosWithOldSign = getPhotosWhereSignIsVisible(
            oldSign, neighboringPhotos);
        if (photosWithOldSign.empty()) {
            // oldSign is not visible from current track
            continue;
        }

        db::Features oldFeaturesOfOldSign = loadFeaturesOfSign(poolHolder,
                                                               oldSign.id());
        if (oldFeaturesOfOldSign.size() < 2
            || mostPhotosAreSideDirected(oldFeaturesOfOldSign))
        {
            continue;
        }

        if (featureAfterDateExist(oldFeaturesOfOldSign,
                                  photosWithOldSign.front().timestamp() - std::chrono::hours(24 * 30)))
        {
            // oldSign was captured this month
            continue;
        }

        INFO() << "Outdated sign " << oldSign.type();
        INFO() << "new photos of the sign position:";
        for (const auto& photo : photosWithOldSign) {
            INFO() << "https://npro.maps.yandex.ru/#!/mrc/" << photo.id();
        }
        INFO() << "old photos of the sign:";
        for (const auto& feature : oldFeaturesOfOldSign) {
            INFO() << "https://npro.maps.yandex.ru/#!/mrc/" << feature.id();
        }
        INFO() << " ";

        outdatedSigns.push_back(oldSign);
    }
    INFO() << "FOUND " << outdatedSigns.size() << " photos";
    return outdatedSigns;
}

db::Signs findOutdatedSigns(wiki::common::PoolHolder& poolHolder,
                            const Photos& newPhotos,
                            const Signs& newSigns)
{
    db::Features newFeatures;
    newFeatures.reserve(newPhotos.size());
    for (const auto& photo : newPhotos) {
        db::Feature feature(photo.featureId);
        feature.setTimestamp(photo.timestamp);
        feature.setMercatorPos(photo.mercatorPos);
        feature.setHeading(photo.heading);
        newFeatures.push_back(feature);
    }

    return findOutdatedSigns(poolHolder, newFeatures, newSigns);
}

} // namespace maps::mrc::tracks_with_sensors
