#include "matcher.h"

#include "types.h"

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/graph_matcher_adapter/include/graph_matcher_adapter.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/assignment_recording_report_gateway.h>
#include <maps/libs/geolib/include/conversion.h>


namespace maps::mrc::tracks_with_sensors {

namespace {

const double RIGHT_HAND_TRAFFIC_SHIFT_METERS = 1.0;

maps::mrc::adapters::TrackSegments geoToMercator(
    maps::mrc::adapters::TrackSegments matchedTrack)
{
    for (auto& segment : matchedTrack) {
        segment.segment = geolib3::Segment2(
            geolib3::geoPoint2Mercator(segment.segment.start()),
            geolib3::geoPoint2Mercator(segment.segment.end()));
    }
    return matchedTrack;
}

// Shifts track to the right side of the road
maps::mrc::adapters::TrackSegments equidistant(
    maps::mrc::adapters::TrackSegments matchedTrack,
    double distanceMeters)
{
    for (auto& segment : matchedTrack) {
        double mercDistance = geolib3::toMercatorUnits(distanceMeters,
                                                       segment.segment.start());
        segment.segment = geolib3::equidistant(
            geolib3::Polyline2(segment.segment),
            mercDistance,
            geolib3::Orientation::Clockwise).segmentAt(0);
    }
    return matchedTrack;
}

db::TrackPoints toTrackPoints(const maps::mrc::adapters::TrackSegments& matchedTrack) {
    db::TrackPoints trackPoints;
    for (const auto& segment : matchedTrack) {
        auto segmentDuration = std::chrono::duration_cast<std::chrono::milliseconds>(
                segment.endTime - segment.startTime);
        double speed = geolib3::toMeters(geolib3::length(segment.segment),
                                         segment.segment.start())
            / (segmentDuration.count() / 1000.0);
        geolib3::Heading heading = geolib3::Direction2(
            segment.segment).heading();

        db::TrackPoint trackPoint;
        trackPoint.setHeading(heading);
        trackPoint.setSpeedMetersPerSec(speed);

        trackPoint.setMercatorPos(segment.segment.start());
        trackPoint.setTimestamp(segment.startTime);
        trackPoints.push_back(trackPoint);

        trackPoint.setMercatorPos(segment.segment.end());
        trackPoint.setTimestamp(segment.endTime);
        trackPoints.push_back(std::move(trackPoint));
    }
    return trackPoints;
}

} // anonymous namespace

db::TrackPoints getMatchedTrack(const db::TrackPoints& trackPoints,
                                const std::optional<std::string>& graphDir)
{
    adapters::TrackSegments matchedTrack;
    if (graphDir) {
        maps::mrc::adapters::GraphMatcherAdapter matcher(*graphDir);
        matchedTrack = matcher.match(trackPoints);
    } else {
        maps::mrc::adapters::GraphMatcherAdapter matcher;
        matchedTrack = matcher.match(trackPoints);
    }
    matchedTrack = geoToMercator(std::move(matchedTrack));
    matchedTrack = equidistant(std::move(matchedTrack),
                               RIGHT_HAND_TRAFFIC_SHIFT_METERS);
    return toTrackPoints(matchedTrack);
}

} // namespace maps::mrc::tracks_with_sensors
