#include "sensors_loader.h"

#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/pb_stream2/reader.h>
#include <yandex/maps/proto/offline_recording/record.pb.h>
#include <yandex/maps/proto/offline_recording/log_event.pb.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/assignment_recording_report_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/include/position_improvment.h>

#include <vector>
#include <string>

namespace precording = yandex::maps::proto::offline::recording;
namespace ugc = maps::mrc::db::ugc;

namespace maps::mrc::tracks_with_sensors {

namespace {

const TString SENSORS_VERSION = "sensors3";
const TString OLD_SENSORS_VERSION_1 = "sensors";
const TString OLD_SENSORS_VERSION_2 = "sensors2";
const TString SENSOR_TYPE_ACCELEROMETER = "accel";
const TString SENSOR_TYPE_GYROSCOPE = "gyro";

struct SensorRecord {
    TString type;
    double x;
    double y;
    double z;
    chrono::TimePoint time;
};
using SensorRecords = std::vector<SensorRecord>;

std::vector<ugc::AssignmentRecordingReport> loadReports(
    wiki::common::PoolHolder& poolHolder,
    db::TId assignmentId)
{
   auto txn = poolHolder.pool().slaveTransaction();
   ugc::AssignmentRecordingReportGateway gtw(*txn);
   return gtw.load(
       ugc::table::AssignmentRecordingReport::assignmentId.equals(assignmentId));
}

SensorRecord extractSensorRecord(
    const precording::log_event::EventRecord& eventRecord)
{
    double x, y, z;
    chrono::TimePoint time;
    bool hasX, hasY, hasZ, hasTime;
    hasX = hasY = hasZ = hasTime = false;
    for (int i = 0; i < eventRecord.params_size(); i++) {
        if (eventRecord.params(i).event() == "x") {
            x = std::stod(eventRecord.params(i).value());
            hasX = true;
        } else if (eventRecord.params(i).event() == "y") {
            y = std::stod(eventRecord.params(i).value());
            hasY = true;
        } else if (eventRecord.params(i).event() == "z") {
            z = std::stod(eventRecord.params(i).value());
            hasZ = true;
        } else if (eventRecord.params(i).event() == "absTs") {
            int64_t milliseconds = std::stod(eventRecord.params(i).value());
            time = chrono::TimePoint(
                std::chrono::milliseconds(milliseconds));
            hasTime = true;
        }
    }
    REQUIRE(hasX && hasY && hasZ && hasTime,
            "event record with missed parameters");
    return SensorRecord{eventRecord.event(), x, y, z, time};
}

SensorRecords loadSensorRecords(
    mds::Mds& mdsClient,
    wiki::common::PoolHolder& poolHolder,
    db::TId assignmentId)
{
    std::vector<ugc::AssignmentRecordingReport> reports
        = loadReports(poolHolder, assignmentId);

    SensorRecords sensorRecords;

    for (auto report : reports) {
        std::stringstream data;
        mdsClient.get(report.mdsKey(), data);
        maps::pb_stream2::Reader reader(&data);
        for (auto it = reader.begin(); it != reader.end(); ++it) {
            auto record = it->as<precording::record::Record>();
            if (record.HasExtension(precording::log_event::EVENT_RECORD)) {
                auto& eventRecord = record.GetExtension(precording::log_event::EVENT_RECORD);
                if (eventRecord.event() != SENSOR_TYPE_ACCELEROMETER
                    && eventRecord.event() != SENSOR_TYPE_GYROSCOPE)
                {
                    continue;
                }
                if (eventRecord.component() == OLD_SENSORS_VERSION_1
                    || eventRecord.component() == OLD_SENSORS_VERSION_2)
                {
                    // skip assignments with old sensor versions
                    return {};
                }
                if (eventRecord.component() != SENSORS_VERSION) {
                    continue;
                }
                sensorRecords.push_back(extractSensorRecord(eventRecord));
            }
        }
    }
    return sensorRecords;
}

} // anonymous namespace

SensorEvents loadSensors(
    mds::Mds& mdsClient,
    wiki::common::PoolHolder& poolHolder,
    db::TId assignmentId,
    chrono::TimePoint minTime,
    chrono::TimePoint maxTime)
{
    SensorRecords sensorRecords = loadSensorRecords(
        mdsClient, poolHolder, assignmentId);

    std::sort(sensorRecords.begin(), sensorRecords.end(),
              [](const SensorRecord& lhs, const SensorRecord& rhs) {
                  return lhs.time < rhs.time;
              });

    pos_improvment::GyroscopeEvents gyroEvents;
    pos_improvment::AccelerometerEvents accEvents;

    for (const SensorRecord& sensorRecord : sensorRecords) {
        if (sensorRecord.time < minTime || sensorRecord.time > maxTime) {
            continue;
        }

        if (sensorRecord.type == SENSOR_TYPE_ACCELEROMETER) {
            accEvents.push_back(
                pos_improvment::createAccelerometerEvent(
                    sensorRecord.time,
                    sensorRecord.x,
                    sensorRecord.y,
                    sensorRecord.z));
        } else {
            gyroEvents.push_back(
                pos_improvment::createGyroscopeEvent(
                    sensorRecord.time,
                    sensorRecord.x,
                    sensorRecord.y,
                    sensorRecord.z));
        }
    }

    return {gyroEvents, accEvents};
}

} // namespace maps::mrc::tracks_with_sensors
