#include <maps/wikimap/mapspro/services/mrc/tools/export_hypotheses/lib/hypothesis.h>
#include <maps/wikimap/mapspro/services/mrc/tools/export_hypotheses/lib/sign.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/hypothesis_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/sign_gateway.h>

#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>

#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <yandex/maps/wiki/social/feedback/gateway_ro.h>
#include <yandex/maps/wiki/social/feedback/task_filter.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/range/algorithm_ext/erase.hpp>


namespace fb = maps::wiki::social::feedback;

using MrcConfig = maps::mrc::common::Config;
using WikiConfig = maps::wiki::common::ExtendedXmlDoc;
using PoolHolder = maps::wiki::common::PoolHolder;

namespace maps {
namespace mrc {
namespace export_hypotheses {

namespace {

fb::Tasks loadTasks(const WikiConfig& wikiConfig)
{
    PoolHolder wiki(wikiConfig, "social", "editor");

    auto txn = wiki.pool().masterReadOnlyTransaction();

    return fb::GatewayRO(*txn).tasksByFilter(fb::TaskFilter().source("mrc"));
}

std::set<db::TId> getExistingFeatureIds(
        pqxx::transaction_base& txn,
        const db::SignFeatures& signFeatures)
{
    db::TIds featureIds;
    for (const auto& box: signFeatures) {
        featureIds.push_back(box.featureId());
    }

    featureIds = db::FeatureGateway(txn).loadIds(
        db::table::Feature::id.in(std::move(featureIds))
    );

    return {featureIds.begin(), featureIds.end()};
}

void run(const MrcConfig& mrcConfig, const WikiConfig& wikiConfig)
{
    INFO() << "Starting...";

    const fb::Tasks tasks = loadTasks(wikiConfig);
    INFO() << "Found " << tasks.size() << " mrc tasks";

    PoolHolder mrc(mrcConfig.makePoolHolder());
    auto txn = mrc.pool().masterWriteableTransaction();

    db::SignFeatures newSignFeatures;
    db::Hypotheses hypotheses;

    for (const auto& task: tasks) {
        const auto& attrs = task.attrs();

        if (!attrs.exist(fb::AttrType::SourceContext)) {
            continue;
        }

        const auto& context = attrs.get(fb::AttrType::SourceContext);

        db::Sign sign = createSign(context["sign"]);
        auto signFeatures = createSignFeatures(context["features"]);

        INFO() << "Check feature ids";
        const auto existingFeatureIds = getExistingFeatureIds(*txn, signFeatures);

        boost::remove_erase_if(
            signFeatures,
            [&](const db::SignFeature& signFeature) {
                return !existingFeatureIds.count(signFeature.featureId());
            }
        );

        if (signFeatures.empty()) {
            WARN() << "Skip!";
            continue;
        }

        db::SignGateway(*txn).upsert(sign);
        INFO() << "Create sign with id " << sign.id();

        newSignFeatures.reserve(newSignFeatures.size() + signFeatures.size());
        for (const auto& signFeature: signFeatures) {
            newSignFeatures.emplace_back(signFeature);
            newSignFeatures.back().setSignId(sign.id());
        }

        hypotheses.push_back(
            createHypothesis(context, sign.id(), task.id())
        );
    }

    INFO() << "Save sign features " << newSignFeatures.size();
    db::SignFeatureGateway(*txn).insert(newSignFeatures);

    INFO() << "Save hypotheses " << hypotheses.size();
    db::HypothesisGateway(*txn).upsert(hypotheses);

    txn->commit();

    INFO() << "Finish!";
}

} // namepspace

} //namespace export_hypotheses
} //namespace mrc
} //namespace maps

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser;
    auto mrcConfigPath = parser.string("mrc-config")
                            .help("path to mrc config");

    auto wikiConfigPath = parser.string("wiki-config")
            .defaultValue("/etc/yandex/maps/wiki/services/services.xml")
            .help("Path to services config for wikimap");

    parser.parse(argc, const_cast<char**>(argv));

    auto mrcConfig = maps::mrc::common::templateConfigFromCmdPath(mrcConfigPath);

    WikiConfig wikiConfig(wikiConfigPath);

    maps::mrc::export_hypotheses::run(mrcConfig, wikiConfig);

    return EXIT_SUCCESS;
} catch (const maps::Exception& ex) {
    FATAL() << "Filed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Failed: " << ex.what();
    return EXIT_FAILURE;
}
