#include "hypothesis.h"

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/serialization.h>

#include <set>
#include <string>


namespace maps {
namespace mrc {
namespace export_hypotheses {

namespace {

json::Value createHypothesisContext(const json::Value& sourceContext)
{
    const static std::set<std::string> DEFAULT_FIELDS {
        "id",
        "features",
        "revisionId",
        "sign",
        "type",
    };

    json::repr::ObjectRepr context;

    for (const auto& field: sourceContext.fields()) {
        if (!DEFAULT_FIELDS.count(field)) {
            context.emplace(field, sourceContext[field]);
        }
    }

    return json::Value{context};
}

} // namespace

db::Hypothesis createHypothesis(
        const json::Value& sourceContext,
        db::TId signId,
        db::TId feedbackTaskId)
{
    db::TId commitId = 0;
    db::TId objectId = 0;

    if (sourceContext["revisionId"].exists()) {
        const std::string revisionId = sourceContext["revisionId"].as<std::string>();

        const size_t index = revisionId.find(':');

        commitId = std::stoull(revisionId.substr(0, index));
        objectId = std::stoull(revisionId.substr(index + 1));
    }

    const geolib3::Point2 position = geolib3::geoPoint2Mercator(
        geolib3::readGeojson<geolib3::Point2>(sourceContext["sign"]["geometry"])
    );

    db::Hypothesis hypothesis {
        boost::lexical_cast<db::HypothesisType>(sourceContext["type"].as<std::string>()),
        position,
        objectId,
        commitId,
        signId
    };

    hypothesis.setContext(createHypothesisContext(sourceContext));
    hypothesis.setFeedbackTaskId(feedbackTaskId);

    return hypothesis;
}

} //namespace export_hypotheses
} //namespace mrc
} //namespace maps
