#include <maps/wikimap/mapspro/services/autocart/libs/geometry/include/hex_wkb.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/exif.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <set>


using namespace maps::mrc;
using namespace NYT;

namespace {

struct CompareFeature {
    bool operator() (const db::Feature& lhs, const db::Feature& rhs) const {
        return lhs.id() < rhs.id();
    }
};

typedef std::set<db::Feature, CompareFeature> FeatureSet;

std::set<db::TId> loadFeatureIds(const std::string& path) {
    std::set<db::TId> results;
    std::ifstream ifs(path);
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        if ('#' == line[0])
            continue;
        results.insert(std::stoi(line));
    }
    return results;
}

/*
    Это не очень честный алгоритм склейки, и он зависит от порядка точек в points
    но мы не предполагаем здесь какой-то сложной специфики и хотим просто избавиться
    от дублирования близколежащих точек
*/
std::vector<maps::geolib3::Point2> joinNearest(std::vector<maps::geolib3::Point2> points) {
    constexpr double RADIUS_METERS = 1.;
    if (points.size() <= 1) {
        return points;
    }
    for (size_t i = 0; i < points.size() - 1; i++) {
        maps::geolib3::Point2& pt = points[i];
        const double radius = maps::geolib3::toMercatorUnits(RADIUS_METERS, pt);
        for (size_t j = i + 1; j < points.size(); j++) {
            if (maps::geolib3::distance(pt, points[j]) > radius) {
                continue;
            }
            const maps::geolib3::Vector2 shift((points[j].x() - pt.x()) / 2., (points[j].y() - pt.y()) / 2.);
            pt += shift;
            points.erase(points.begin() + j);
            j--;
        }
    }
    return points;
}


class GeometrySearcher {
    typedef maps::geolib3::StaticGeometrySearcher<maps::geolib3::Segment2, int> InternalSearcher;
public:
    void insert(const maps::geolib3::MultiPolygon2 mpgn) {
        for (size_t pgnIdx = 0; pgnIdx < mpgn.polygonsNumber(); pgnIdx++) {
            insertPolygon(mpgn.polygonAt(pgnIdx));
        }
    }
    void build() {
        INFO() << "Build static geometry searcher, segments count: " << segments_.size();
        searcher_.build();
    }
    std::vector<maps::geolib3::Point2> getIntersections(const maps::geolib3::Polyline2& mercatorPolyline) const {
        std::vector<maps::geolib3::Point2> points;
        const InternalSearcher::SearchResult internal = searcher_.find(mercatorPolyline.boundingBox());
        for (auto itr = internal.first; itr != internal.second; itr++) {
            for (size_t i = 0; i < mercatorPolyline.segmentsNumber(); i++) {
                std::vector<maps::geolib3::Point2> temp = maps::geolib3::intersection(mercatorPolyline.segmentAt(i), itr->geometry());
                if (temp.empty()) {
                    continue;
                } else if (1 == temp.size()) {
                    points.emplace_back(temp[0]);
                } else if (2 == temp.size()) {
                    points.emplace_back((temp[0].x() + temp[1].x()) / 2., (temp[0].y() + temp[1].y()) / 2.);
                }
            }
        }
        return joinNearest(points);
    }
private:
    InternalSearcher searcher_;
    std::list<maps::geolib3::Segment2> segments_;
    void insertSegment(const maps::geolib3::Segment2 segment) {
        segments_.emplace_back(segment);
        searcher_.insert(&segments_.back(), -1);
    }

    void insertRing(const maps::geolib3::LinearRing2 ring) {
        const size_t number = ring.segmentsNumber();
        for (size_t i = 0; i < number; i++) {
            insertSegment(ring.segmentAt(i));
        }
    }

    void insertPolygon(const maps::geolib3::Polygon2 pgn) {
        insertRing(pgn.exteriorRing());
        const size_t number = pgn.interiorRingsNumber();
        for (size_t i = 0; i < number; i++) {
            insertRing(pgn.interiorRingAt(i));
        }
    }
};

void fillSearcher(GeometrySearcher& searcher, IClientPtr client, const TString& tablePath) {
    TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(tablePath);
    for (int processedItems = 0; reader->IsValid(); reader->Next(), processedItems++) {
        if ((processedItems + 1) % 100000 == 0) {
            INFO() << "Processed " << (processedItems + 1) << " items";
        }
        const TNode& inpRow = reader->GetRow();
        searcher.insert(convertGeodeticToMercator(maps::wiki::autocart::hexWKBToMultiPolygon(inpRow["geom"].AsString())));
    };
    searcher.build();
}

std::vector<maps::geolib3::Point2> getIntersection(const GeometrySearcher& searcher, IClientPtr client, const TString& tablePath) {
    std::vector<maps::geolib3::Point2> points;
    TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(tablePath);
    for (int processedItems = 0; reader->IsValid(); reader->Next(), processedItems++) {
        if ((processedItems + 1) % 100000 == 0) {
            INFO() << "Processed " << (processedItems + 1) << " road segments, found " << points.size() << " intersection points";
        }
        const TNode& inpRow = reader->GetRow();
        const maps::geolib3::Polyline2 pln = convertGeodeticToMercator(maps::wiki::autocart::hexWKBToPolyline(inpRow["geom"].AsString()));
        std::vector<maps::geolib3::Point2> temp = searcher.getIntersections(pln);
        if (!temp.empty()) {
            points.insert(points.end(), temp.begin(), temp.end());
        }
    };
    return points;
}

void filter(db::Features& features, const std::set<db::TId>& filteredFids) {
    features.erase(
        std::remove_if(features.begin(), features.end(),
            [&](const db::Feature& feature) {
                return (0 < filteredFids.count(feature.id()));
            }
        ),
        features.end()
    );
}

std::string getFeatureUrl(const db::Feature& f) {
    return ("http://storage-int.mds.yandex.net/get-maps_mrc/" + f.mdsGroupId() + "/" + f.mdsPath()).c_str();
}

void saveToFile(const FeatureSet& features, const std::string& path) {
    constexpr db::TId MIN_FEATURE_ID = 52500000; // это где-то начало 2019 года, как мне кажется там качество лучше
    std::ofstream ofs(path);
    for (const db::Feature& feature : features) {
        if (feature.id() < MIN_FEATURE_ID) {
            continue;
        }
        ofs << feature.id() << " " << getFeatureUrl(feature) << " " << (int)feature.orientation() << std::endl;
    }
}

FeatureSet loadNearestFeatures(
    maps::pgpool3::Pool& pool,
    const std::vector<maps::geolib3::Point2>& points,
    const std::set<db::TId>& filterFids
)
{
    constexpr double BBOX_SIZE_METERS = 200.;
    constexpr double JOIN_RADIUS_METERS = 20.;

    std::vector<bool> reworked(points.size(), false);
    maps::geolib3::StaticGeometrySearcher<maps::geolib3::Point2, size_t> searcher;
    for (size_t i = 0; i < points.size(); i++) {
        searcher.insert(&points[i], i);
    }
    searcher.build();

    int requestCount = 0;
    FeatureSet result;
    auto txn = pool.slaveTransaction();
    for (size_t i = 0; i < points.size(); i++) {
        if (reworked[i]) {
            continue;
        }
        reworked[i] = true;

        const maps::geolib3::Point2& pt = points[i];
        const double joinSize = 2. * maps::geolib3::toMercatorUnits(JOIN_RADIUS_METERS, pt);
        const double bboxSize = maps::geolib3::toMercatorUnits(BBOX_SIZE_METERS, pt);

        maps::geolib3::BoundingBox bbox(pt, bboxSize, bboxSize);
        const auto found = searcher.find(maps::geolib3::BoundingBox(pt, joinSize, joinSize));
        for (auto itr = found.first; itr != found.second; itr++) {
            const size_t ptIdx = itr->value();
            if (reworked[ptIdx]) {
                continue;
            }
            bbox = maps::geolib3::expand(bbox, maps::geolib3::BoundingBox(points[ptIdx], bboxSize, bboxSize));
            reworked[ptIdx] = true;
        }

        db::Features nearest = db::FeatureGateway{*txn}.load(db::table::Feature::isPublished && db::table::Feature::pos.intersects(bbox));
        filter(nearest, filterFids);
        result.insert(nearest.begin(), nearest.end());

        requestCount++;
        if (requestCount % 10000 == 0) {
            INFO() << "Processed " << requestCount << " items, found "  << result.size() << " features";
        }
    }

    INFO() << "Request count: " << requestCount;
    INFO() << "Features count:  " << result.size();

    return result;
}


} //namespace

int main(int argc, const char** argv) try {
    static const TString YT_PROXY = "hahn";

    maps::cmdline::Parser parser("Extract features near intersection of segments from two sets of geometry.");

    maps::cmdline::Option<std::string> mrcConfigPath = parser.string("mrc-config")
        .help("Path to mrc config");

    maps::cmdline::Option<std::string> secretVersion = parser.string("secret-version")
        .help("version for secrets from yav.yandex-team.ru");

    maps::cmdline::Option<std::string> geomYTPath1 = parser.string("input1")
        .help("YT table with first set of geometries");

    maps::cmdline::Option<std::string> geomYTPath2 = parser.string("input2")
        .help("YT table with second set of geometries");

    maps::cmdline::Option<std::string> filterPath = parser.string("filter")
        .help("List of features ids will erased from results");

    maps::cmdline::Option<std::string> outputPath = parser.string("output")
        .help("Output path for feature id, url, orientation");

    parser.parse(argc, const_cast<char**>(argv));

    maps::mrc::common::Config mrcConfig =
        secretVersion.defined()
        ? maps::mrc::common::Config(maps::vault_boy::loadContextWithYaVault(secretVersion), mrcConfigPath)
        : maps::mrc::common::templateConfigFromCmdPath(mrcConfigPath);
    maps::wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());

    std::set<db::TId> filterFids = loadFeatureIds(filterPath);
    INFO() << "Feature ids for filter: " << filterFids.size();

    INFO() << "Connecting to yt::" << YT_PROXY;
    IClientPtr client = CreateClient(YT_PROXY);

    GeometrySearcher searcher;
    fillSearcher(searcher, client, geomYTPath1.c_str());

    std::vector<maps::geolib3::Point2> points = getIntersection(searcher, client, geomYTPath2.c_str());
    INFO() << "Found points: " << points.size();

    FeatureSet features = loadNearestFeatures(mrc.pool(), points, filterFids);
    saveToFile(features, outputPath);

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
