#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point_gateway.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/config/include/config.h>
#include <maps/tools/easyview/lib/io/include/io.h>
#include <library/cpp/yson/node/node_io.h>
#include <library/cpp/iterator/enumerate.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <tuple>

namespace ev = maps::tools::easyview;
namespace db = maps::mrc::db;

const std::string COLORMAP_RESOURCE = "cmap.txt";

auto featuresSourceIdDateFilter(
    const std::string& sourceId,
    const std::string& day
) {
    auto dayStart = maps::chrono::parseSqlDateTime(day + " 00:00:00");
    auto dayEnd = maps::chrono::parseSqlDateTime(day + " 23:59:59");
    return db::table::Feature::sourceId.equals(sourceId) &&
           db::table::Feature::date.between(dayStart, dayEnd) &&
           db::table::Feature::pos.isNotNull();
}

auto trackPointsSourceIdDateFilter(
    const std::string& sourceId,
    const std::string& day
) {
    auto dayStart = maps::chrono::parseSqlDateTime(day + " 00:00:00");
    auto dayEnd = maps::chrono::parseSqlDateTime(day + " 23:59:59");
    return db::table::TrackPoint::sourceId.equals(sourceId) &&
           db::table::TrackPoint::timestamp.between(dayStart, dayEnd);
}

class ColorMap {
public:
    ColorMap(double alpha = 1.0) {
        std::stringstream colors(maps::config::readConfigFile(COLORMAP_RESOURCE));
        double r, g, b;

        while (colors >> r >> g >> b) {
            colors_.push_back(ev::Color::rgba(r, g, b, alpha));
        }
    }

    ev::Color operator()(double ratio) const {
        ratio = std::fmin(ratio, 1.0);
        ratio = std::fmax(ratio, 0.0);

        int idx = static_cast<int>(colors_.size() * ratio);
        return colors_[idx];
    }

private:
    std::vector<ev::Color> colors_;
};

class EasyviewWriter {
public:
    EasyviewWriter(
        std::string outputDir,
        ev::Color featureBorderColor,
        ev::Color trackPointBorderColor,
        ColorMap colorMap
    )
        : outputDir_(std::move(outputDir))
        , featureBorderColor_(std::move(featureBorderColor))
        , trackPointBorderColor_(std::move(trackPointBorderColor))
        , colorMap_(std::move(colorMap))
    {}

    void operator()(
        const db::Features& features,
        const db::TrackPoints& trackPoints,
        const std::string& sourceId,
        const std::string& day
    ) {
        if (not features.size()) {
            return;
        }

        std::string fileName = outputDir_ + "/" + sourceId + "_" + day;
        std::ofstream file(fileName, std::ofstream::app);

        for (const auto [i, feature] : Enumerate(features)) {
            file << formatFeature(feature, day, 1.0 * i / features.size()) << std::endl;
        }

        for (const auto [i, trackPoint] : Enumerate(trackPoints)) {
            file << formatTrackPoint(trackPoint, 1.0 * i / trackPoints.size()) << std::endl;
        }
    }

    std::string formatFeature(
        const db::Feature& feature,
        const std::string& day,
        double normalizedIdx
    ) {
        std::string button =
            "<button onclick=markAsBad('" + feature.sourceId() +
            "','" + day + "','" +
            std::to_string(feature.id()) + "')>mark_bad</button>";

        std::string img_link =
            "<a href='https://n.maps.yandex.ru/#!/mrc/" +
            std::to_string(feature.id()) +
            "' target='blank'>photos</a>";

        static constexpr size_t nano = 1e9;
        const size_t timestamp = feature.timestamp().time_since_epoch().count() / nano;

        NYT::TNode node = NYT::TNode::CreateMap()
            ("feature_id", feature.id())
            ("timestamp", timestamp)
            (ev::yson::POINTSTYLE, ev::yson::pointStyle(colorMap_(normalizedIdx), featureBorderColor_, 4.0))
            ("lon", feature.geodeticPos().x())
            ("lat", feature.geodeticPos().y())
            ("mark_bad", button.c_str())
            ("img", img_link.c_str());

        return NYT::NodeToCanonicalYsonString(node);
    }

    std::string formatTrackPoint(
        const db::TrackPoint& trackPoint,
        double normalizedIdx
    ) {
        static constexpr size_t nano = 1e9;
        const size_t timestamp = trackPoint.timestamp().time_since_epoch().count() / nano;

        NYT::TNode node = NYT::TNode::CreateMap()
            ("feature_id", trackPoint.id())
            ("timestamp", timestamp)
            (ev::yson::POINTSTYLE, ev::yson::pointStyle(colorMap_(normalizedIdx), trackPointBorderColor_, 4.0))
            ("lon", trackPoint.geodeticPos().x())
            ("lat", trackPoint.geodeticPos().y())
            ("accuracy", trackPoint.accuracyMeters() ? *trackPoint.accuracyMeters() : -1);

        return NYT::NodeToCanonicalYsonString(node);
    }

private:
    std::string outputDir_;
    ev::Color featureBorderColor_;
    ev::Color trackPointBorderColor_;
    ColorMap colorMap_;
};


int main(int argc, char **argv) {
    maps::cmdline::Parser parser("Download features and tracks gps data");
    auto mrcConfigPath = parser.string("mrc-config")
        .help("mrc config path")
        .required();
    auto secretVersion = parser.string("secret-version")
        .help("secret version from yav")
        .required();
    auto sourceIdsFilePath = parser.string("source-ids")
        .help("path to file with source ids and dates")
        .required();
    auto outputDir = parser.string("output-dir")
        .help("output directory (where files with gps will be created)")
        .required();

    parser.parse(argc, argv);

    std::ifstream sourceIdsFile(sourceIdsFilePath);
    std::string sourceId, day;

    const auto mrcConfig =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);

    auto poolHolder = mrcConfig.makePoolHolder();
    ColorMap colorMap{};
    EasyviewWriter writer = EasyviewWriter(
        outputDir,
        ev::Color::rgba(1.0, 0.0, 0.0, 0.3),
        ev::Color::rgba(0.0, 1.0, 0.0, 0.3),
        colorMap
    );

    while (sourceIdsFile >> sourceId >> day) {
        auto txn = poolHolder.pool().slaveTransaction();
        maps::mrc::db::Features features = maps::mrc::db::FeatureGateway(*txn).load(featuresSourceIdDateFilter(sourceId, day));
        maps::mrc::db::TrackPoints tracks = maps::mrc::db::TrackPointGateway(*txn).load(trackPointsSourceIdDateFilter(sourceId, day));
        std::sort(
            features.begin(),
            features.end(),
            [](const auto& lhs, const auto& rhs) {
                return lhs.timestamp() < rhs.timestamp();
            }
        );
        std::sort(
            tracks.begin(),
            tracks.end(),
            [](const auto& lhs, const auto& rhs) {
                return lhs.timestamp() < rhs.timestamp();
            }
        );
        writer(features, tracks, sourceId, day);
    }
}
