#include "context.h"
#include "magic_strings.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/metadata_gateway.h>

#include <maps/libs/sql_chemistry/include/system_information.h>

#include <chrono>

namespace maps {
namespace mrc {
namespace features_deleter {

Context::Context(const maps::mrc::common::Config& cfg,
                 const std::string& graphFolder)
    : poolHolder_(cfg.makePoolHolder())
    , mds_(cfg.makeMdsClient())
    , graph_(graphFolder)
{
}

pgpool3::TransactionHandle Context::masterReadOnlyTransaction()
{
    return poolHolder_.pool().masterReadOnlyTransaction();
}

pgpool3::Pool& Context::pool()
{
    return poolHolder_.pool();
}

db::TIds Context::loadFeaturesIds()
{
    auto txn = poolHolder_.pool().slaveTransaction();
    db::FeatureGateway gtw(*txn);
    return gtw.loadIds();
}

db::Features Context::loadFeaturesByIds(const db::TIds& featureIds)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    db::FeatureGateway gtw(*txn);
    return gtw.loadByIds(featureIds);
}

db::Features Context::loadFeatures(const sql_chemistry::Filter& filter)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    db::FeatureGateway gtw(*txn);
    return gtw.load(filter);
}

void Context::unpublishFeatures(db::Features& features)
{
    auto txn = poolHolder_.pool().masterWriteableTransaction();

    for (auto& feature : features) {
        feature.setAutomaticShouldBePublished(false);
    }
    db::FeatureGateway{*txn}.update(features, db::UpdateFeatureTxn::Yes);
    txn->commit();
}

void Context::saveMetadata(size_t deletedCount)
{
    auto txn = poolHolder_.pool().masterWriteableTransaction();
    db::MetadataGateway gtw{*txn};

    const auto timestamp =  std::chrono::duration_cast<std::chrono::seconds>(
        std::chrono::system_clock::now().time_since_epoch()
    );

    gtw.upsertByKey(
        metadataKey(TIMESTAMP_KEY),
        timestamp.count()
    );
    gtw.upsertByKey(metadataKey(DELETED_COUNT_KEY), deletedCount);
    txn->commit();
}

Graph& Context::graph() { return graph_; }

} // features_deleter
} // mrc
} // maps
