#include "context.h"
#include "find_garbage.h"
#include "remove_features.h"

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool3utils/include/yandex/maps/pgpool3utils/pg_advisory_mutex.h>

#include <memory>
#include <mutex>

namespace mrc = maps::mrc;

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser(
        "Tool for deleting needless and bad-quality MRC photos.");
    auto configPath = parser.string("config").help("path to configuration");
    auto graphFolder
        = parser.string("graph-folder")
            .help("graph data will be taken from a specified directory");
    auto isDryRun = parser.flag("dry-run")
        .help("don't modify database and MDS")
        .defaultValue(false);
    auto syslog = parser.string("syslog-tag")
        .help("redirect log output to syslog with given tag");
    parser.parse(argc, argv);

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }

    auto cfg = maps::mrc::common::templateConfigFromCmdPath(configPath);
    mrc::features_deleter::Context ctx(cfg, graphFolder);
    maps::pgp3utils::PgAdvisoryXactMutex mutex(
        ctx.pool(),
        static_cast<int64_t>(maps::mrc::common::LockId::FeaturesDeleter)
                                              );
    if (!mutex.try_lock()) {
        INFO() << "Another process is ongoing";
        return 0;
    }

    auto featureIds = findGarbageFeatureIds(ctx);
    if (!isDryRun) {
        ctx.saveMetadata(unpublishFeaturesByIds(ctx, featureIds));
    }
    return 0;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return 1;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return 1;
}
catch (...) {
    ERROR() << "unknown error";
    return 1;
}
