#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/common/include/exception.h>

#include <maps/wikimap/mapspro/services/mrc/libs/carsegm/include/carsegm.h>
#include <maps/wikimap/mapspro/services/mrc/libs/privacy_detector/include/privacy_detector_faster_rcnn.h>

#include <mapreduce/yt/interface/client.h>
//#include <mapreduce/yt/util/temp_table.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <fstream>

namespace {

static const TString COLUMN_NAME_DATE       = "date";
static const TString COLUMN_NAME_FEATURE_ID = "feature_id";
static const TString COLUMN_NAME_HEADING    = "heading";
static const TString COLUMN_NAME_IMAGE      = "image";
static const TString COLUMN_NAME_LAT        = "lat";
static const TString COLUMN_NAME_LON        = "lon";
static const TString COLUMN_NAME_OBJECTS    = "objects";

NYT::TNode createSchema() {
    return NYT::TNode::CreateList()
                .Add(NYT::TNode()("name", COLUMN_NAME_DATE)("type", "string"))
                .Add(NYT::TNode()("name", COLUMN_NAME_FEATURE_ID)("type", "int64"))
                .Add(NYT::TNode()("name", COLUMN_NAME_HEADING)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_IMAGE)("type", "string"))
                .Add(NYT::TNode()("name", COLUMN_NAME_LAT)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_LON)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_OBJECTS)("type", "any"));
}

cv::Mat decodeImage(const TString& encimageStr) {
    std::vector<std::uint8_t> encimage(Base64DecodeBufSize(encimageStr.length()));
    size_t encimageSize = Base64Decode(encimage.data(), encimageStr.begin(), encimageStr.end());
    encimage.resize(encimageSize);
    return cv::imdecode(encimage, cv::IMREAD_COLOR + cv::IMREAD_IGNORE_ORIENTATION);
}

TString encodeImage(const cv::Mat& image) {
    std::vector<uint8_t> encImage;
    cv::imencode(".jpg", image, encImage);
    return maps::base64Encode(std::string(encImage.begin(), encImage.end())).c_str();
}

cv::Rect NodeToRect(const NYT::TNode& node) {
    cv::Rect rc;
    const TVector<NYT::TNode>& rcNode = node.AsList();
    const int64_t x1 = rcNode[0].AsList()[0].AsInt64();
    const int64_t y1 = rcNode[0].AsList()[1].AsInt64();

    const int64_t x2 = rcNode[1].AsList()[0].AsInt64();
    const int64_t y2 = rcNode[1].AsList()[1].AsInt64();

    rc.x = std::min((int)x1, (int)x2);
    rc.y = std::min((int)y1, (int)y2);
    rc.width = (int)x2 - (int)x1;
    rc.height = (int)y2 - (int)y1;
    return rc;
}

std::vector<cv::Rect> readObjectsRects(const NYT::TNode& node) {
    static const TString ITEM_NAME_BBOX         = "bbox";
    std::vector<cv::Rect> result;
    const TVector<NYT::TNode>& objectList = node.AsList();
    for (const NYT::TNode& objectNode : objectList) {
        result.emplace_back(NodeToRect(objectNode[ITEM_NAME_BBOX]));
    }
    return result;
}

constexpr size_t MiB = 1024 * 1024;
constexpr size_t GiB = 1024 * MiB;

NYT::TNode createOperationSpec(const TString& operationType, bool useGpu) {
    NYT::TNode operationSpec = NYT::TNode::CreateMap()
        ("title", "House numbers convert dataset")
        ("scheduling_tag_filter", "!testing");

    const TString GPU_POOL_TREES = "gpu_geforce_1080ti";

    if (useGpu) {
        operationSpec
        ("pool_trees", NYT::TNode::CreateList().Add(GPU_POOL_TREES))
        ("scheduling_options_per_pool_tree", NYT::TNode::CreateMap()
            (GPU_POOL_TREES, NYT::TNode::CreateMap()("pool", "research_gpu"))
        )
        (operationType, NYT::TNode::CreateMap()
            ("memory_limit", 16 * GiB)
            ("gpu_limit", 1)
            ("layer_paths",  NYT::TNode::CreateList()
                            .Add("//porto_layers/delta/gpu/cuda/10.1")
                            .Add("//porto_layers/delta/gpu/driver/418.67")
                            .Add("//porto_layers/base/bionic/porto_layer_search_ubuntu_bionic_app_lastest.tar.gz")
            )
        );
    } else {
        operationSpec
        (operationType, NYT::TNode::CreateMap()
            ("cpu_limit", 4)
            ("memory_limit", 16 * GiB)
            ("memory_reserve_factor", 0.6)
        );
    }
    return operationSpec;
}


// YT Mappers
class TConverterMapper
    : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>>  {
public:
    Y_SAVELOAD_JOB(blurCars_, blurPrivate_);

    TConverterMapper() = default;

    TConverterMapper(bool blurCars, bool blurPrivate)
        : blurCars_(blurCars)
        , blurPrivate_(blurPrivate)
    { }

    void Do(NYT::TTableReader<NYT::TNode>* reader, NYT::TTableWriter<NYT::TNode>* writer) override {
        INFO() << "Start detection ... ";
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode &inpRow = reader->GetRow();
            cv::Mat image = decodeImage(inpRow[COLUMN_NAME_IMAGE].AsString());

            INFO() << "Feature Id: " << inpRow[COLUMN_NAME_FEATURE_ID].AsInt64();
            std::vector<cv::Rect> rcObjects = readObjectsRects(inpRow[COLUMN_NAME_OBJECTS]);
            cv::Mat mask;
            if (blurCars_) {
                mask = getCarsMask(image);
            } else {
                mask = cv::Mat::zeros(image.size(), CV_8UC1);
            }

            if (blurPrivate_) {
                updateMaskByPrivate(image, mask);
            }

            removeRectsFromMask(rcObjects, mask);
            cv::Mat temp[] = { mask, mask, mask };
            cv::merge(temp, 3, mask);
            cv::Mat outImage = blurImage(image, mask);

            NYT::TNode outRow;
            outRow(COLUMN_NAME_FEATURE_ID, inpRow[COLUMN_NAME_FEATURE_ID].AsInt64())
                  (COLUMN_NAME_DATE, inpRow[COLUMN_NAME_DATE].AsString())
                  (COLUMN_NAME_HEADING, inpRow[COLUMN_NAME_HEADING].AsDouble())
                  (COLUMN_NAME_IMAGE, encodeImage(outImage))
                  (COLUMN_NAME_LAT, inpRow[COLUMN_NAME_LAT].AsDouble())
                  (COLUMN_NAME_LON, inpRow[COLUMN_NAME_LON].AsDouble())
                  (COLUMN_NAME_OBJECTS, inpRow[COLUMN_NAME_OBJECTS]);

            writer->AddRow(outRow);
        }
    }
private:
    bool blurCars_;
    bool blurPrivate_;
    maps::mrc::carsegm::CarSegmentator carSegmentator_;
    maps::mrc::privacy_detector::FasterRCNNDetector privacyDetector_;

    cv::Mat getCarsMask(const cv::Mat& image) {
        return carSegmentator_.segment(image);
    }
    void updateMaskByPrivate(const cv::Mat& image, cv::Mat& mask) {
        maps::mrc::privacy_detector::PrivacyImageBoxes boxes = privacyDetector_.detect(image);
        for (size_t i = 0; i < boxes.size(); i++) {
            const cv::Rect& rc = boxes[i].box;
            if (0 < rc.width && 0 < rc.height) {
                mask(rc).setTo(255);
            }
        }
    }
    void removeRectsFromMask(const std::vector<cv::Rect>& rcObjects, cv::Mat& mask) {
        constexpr int BORDER_SIZE = 5;

        for (size_t i = 0; i < rcObjects.size(); i++) {
            const cv::Rect& rc = rcObjects[i];
            cv::Rect rcExp;
            rcExp.x = std::max(rc.x - BORDER_SIZE, 0);
            rcExp.y = std::max(rc.y - BORDER_SIZE, 0);
            rcExp.width = std::min(rc.x + rc.width + BORDER_SIZE, mask.cols) - rcExp.x;
            rcExp.height = std::min(rc.y + rc.height + BORDER_SIZE, mask.rows) - rcExp.y;
            mask(rcExp).setTo(0);
        }
    }
    cv::Mat blurImage(const cv::Mat& image, const cv::Mat& mask) {
        constexpr int KERNEL_SIZE = 21;
        constexpr double SIGMA = 9.0;

        cv::Mat imageBlured;
        cv::GaussianBlur(image, imageBlured, cv::Size(KERNEL_SIZE, KERNEL_SIZE), SIGMA);
        cv::bitwise_and(imageBlured, mask, imageBlured);
        cv::Mat invMask;
        cv::bitwise_not(mask, invMask);
        cv::Mat imageMasked;
        cv::bitwise_and(image, invMask, imageMasked);
        cv::add(imageBlured, imageMasked, imageBlured);
        return imageBlured;
    }
};

REGISTER_MAPPER(TConverterMapper);

} //namespace


int main(int argc, const char** argv) try {
    static const TString YT_PROXY = "hahn";
    static const int JOB_COUNT = 10;

    NYT::Initialize(argc, argv);

    maps::cmdline::Parser parser("Convert house number signs dataset, blur car or/and private data");

    maps::cmdline::Option<std::string> inputYTPath = parser.string("input")
        .required()
        .help("Path to YT table with input dataset");

    maps::cmdline::Option<std::string> outputYTPath = parser.string("output")
        .required()
        .help("Path to YT table for output dataset");

    maps::cmdline::Option<bool> blurCars = parser.flag("blur-cars")
        .help("Blur cars pixels defined by segmentator");

    maps::cmdline::Option<bool> blurPrivate = parser.flag("blur-private")
        .help("Blur private objects found by detector");

    maps::cmdline::Option<bool> useGpu = parser.flag("use-gpu")
        .help("Use GPU");

    parser.parse(argc, const_cast<char**>(argv));

    REQUIRE(blurCars || blurPrivate, "At least one blur-xxx flag should be enabled");

    INFO() << "Connecting to yt::" << YT_PROXY;
    NYT::IClientPtr client = NYT::CreateClient(YT_PROXY);
    INFO() << "Opening table " << inputYTPath;

    client->Create(outputYTPath.c_str(), NYT::NT_TABLE, NYT::TCreateOptions().Attributes(NYT::TNode()("schema", createSchema())));

    client->Map(
        NYT::TMapOperationSpec()
            .AddInput<NYT::TNode>(inputYTPath.c_str())
            .AddOutput<NYT::TNode>(outputYTPath.c_str())
            .JobCount(JOB_COUNT),
        new TConverterMapper(blurCars, blurPrivate),
        NYT::TOperationOptions().Spec(createOperationSpec("mapper", useGpu))
    );
    client->Sort(
        NYT::TSortOperationSpec()
            .AddInput(outputYTPath.c_str())
            .Output(outputYTPath.c_str())
            .SortBy(COLUMN_NAME_FEATURE_ID)
    );
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
