#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/temp_table.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/extended_xml_doc.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/pgpool3_helpers.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/revisionsgateway.h>

#include <maps/wikimap/mapspro/services/mrc/libs/object/include/loader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/revision_loader.h>

#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/thread/pool.h>
#include <util/thread/lfqueue.h>

#include <vector>
#include <string>
#include <fstream>
#include <thread>

using namespace NYT;

namespace {

static const TString YT_PROXY = "hahn";
static const TString YT_TABLE = "//home/maps/core/nmaps/analytics/feedback/db/feedback_latest";
static const TString YT_TEMPORARY_FOLDER = "//tmp/maps_mrc";

static const TString COL_ID         = "id";
static const TString COL_TYPE       = "type";
static const TString COL_SOURCE     = "source";
static const TString COL_RESOLUTION = "resolution";
static const TString COL_WORKFLOW   = "workflow";
static const TString COL_COMMIT_IDS = "commit_ids";
static const TString COL_POSITION   = "position";

static const TString FEEDBACK_TYPE     = "address";
static const TString FEEDBACK_SOURCE   = "mrc";
static const TString FEEDBACK_ACCEPTED = "accepted";

static const TString COLUMN_NAME_FEATURE_ID = "feature_id";
static const TString COLUMN_NAME_LAT        = "lat";
static const TString COLUMN_NAME_LON        = "lon";
static const TString COLUMN_NAME_OBJECTS    = "objects";

static const TString ITEM_NAME_ADDR_NUMBER = "addr_number";
static const TString ITEM_NAME_ADDR_PT_LAT = "addr_pt_lat";
static const TString ITEM_NAME_ADDR_PT_LON = "addr_pt_lon";
static const TString ITEM_NAME_HYP_NUM     = "hypothesis_number";
static const TString ITEM_NAME_BBOX        = "bbox";

static const TString TABLE_ATTR_NAME_COMMIT_ID = "mapCommitID";

static const std::string CAT_ADDRESS_NAME = "cat:addr";

static const std::string ADDRESS_CATEGORY_NAME = "addr";
static const std::string ADDRESS_NAME_CATEGORY_NAME = "addr_nm";
static const std::string RELATION_MASTER_ATTR_NAME = "rel:master";
static const std::string RELATION_SLAVE_ATTR_NAME = "rel:slave";

static const std::string ADDRESS_NAME_NAME_ATTR_NAME = "addr_nm:name";

constexpr double SEARCH_RADIUS_METERS = 75.;
constexpr double EPSILON_METERS = 0.1;
constexpr double POINT_WITHOUT_BUILDING_RADIUS_METERS = 6.875;

constexpr std::chrono::milliseconds THREAD_WAIT_TIMEOUT(1);


std::vector<maps::wiki::revision::DBID> extractIds(const TNode& inpRow) {
    TVector<TNode> listNode = inpRow.AsList();
    std::vector<maps::wiki::revision::DBID> commitIds(listNode.size());
    for (size_t i = 0; i < listNode.size(); i++) {
        commitIds[i] = listNode[i].AsInt64();
    }
    return commitIds;
}

maps::wiki::revision::filters::ProxyFilterExpr createObjectFilterByCommit(
    const std::string &categoryId,
    maps::wiki::revision::DBID commitId)
{
    return  maps::wiki::revision::filters::Attr(categoryId).defined() &&
            maps::wiki::revision::filters::ObjRevAttr::isRegularObject() &&
            maps::wiki::revision::filters::ObjRevAttr::isNotDeleted() &&
            (maps::wiki::revision::filters::ObjRevAttr::commitId() == commitId);
}

maps::wiki::revision::filters::ProxyFilterExpr createObjectFilterByObjectId(uint64_t objectId)
{
    return  maps::wiki::revision::filters::ObjRevAttr::isRegularObject() &&
            maps::wiki::revision::filters::ObjRevAttr::isNotDeleted() &&
            (maps::wiki::revision::filters::ObjRevAttr::objectId() == objectId);
}

maps::wiki::revision::filters::ProxyFilterExpr createRelationFilterByMaster(const uint64_t masterId)
{
    return maps::wiki::revision::filters::ObjRevAttr::isRelation()
        && maps::wiki::revision::filters::ObjRevAttr::isNotDeleted()
        && (maps::wiki::revision::filters::ObjRevAttr::masterObjectId() == masterId);
}

uint64_t getAddressNameObjectID(
    const maps::wiki::revision::Snapshot& snapshot,
    uint64_t addressPointObjectId)
{
    maps::wiki::revision::Revisions relations =
        snapshot.objectRevisionsByFilter(createRelationFilterByMaster(addressPointObjectId));

    for (maps::wiki::revision::Revisions::const_iterator cit = relations.begin();
        cit != relations.end(); cit++) {
        const maps::wiki::revision::ObjectData &data = cit->data();

        if (!data.attributes || !data.relationData)
            continue;

        const auto citMaster = data.attributes->find(RELATION_MASTER_ATTR_NAME);
        if (citMaster == data.attributes->end() || citMaster->second != ADDRESS_CATEGORY_NAME)
            continue;
        const auto citSlave = data.attributes->find(RELATION_SLAVE_ATTR_NAME);
        if (citSlave == data.attributes->end() ||  citSlave->second != ADDRESS_NAME_CATEGORY_NAME)
            continue;
        return data.relationData->slaveObjectId();
    }
    return (uint64_t)-1;
}

std::optional<std::pair<maps::geolib3::Point2, TString> >
    getAddressPoint(
        const maps::wiki::revision::Snapshot& snapshot,
        const std::vector<maps::wiki::revision::DBID>& commitIds)
{
    std::vector<std::pair<uint64_t, maps::geolib3::Point2> > objects;
    for (size_t i = 0; i < commitIds.size(); i++) {
        maps::wiki::revision::Revisions revisions =
            snapshot.objectRevisionsByFilter(
                createObjectFilterByCommit(
                    CAT_ADDRESS_NAME,
                    commitIds[i])
            );
        for (auto cit = revisions.cbegin(); cit != revisions.cend(); cit++) {
            objects.emplace_back(
                    cit->id().objectId(),
                    maps::geolib3::convertMercatorToGeodetic(
                        maps::geolib3::WKB::read<maps::geolib3::Point2>(*(cit->data().geometry)))
            );
        }
    }

    if (1 != objects.size()) {
        return std::nullopt;
    }

    uint64_t addressNameObjectID = getAddressNameObjectID(snapshot, objects[0].first);
    REQUIRE((uint64_t)-1 != addressNameObjectID, "Unable to found address name object id for address point with id " << objects[0].first);
    maps::wiki::revision::Revisions revisions =
        snapshot.objectRevisionsByFilter(createObjectFilterByObjectId(addressNameObjectID));
    REQUIRE(1 == revisions.size(), "Unable to found address name object with id " << addressNameObjectID);
    const maps::wiki::revision::ObjectData& data = revisions.front().data();
    REQUIRE(data.attributes, "Address name object does not contain attributes");
    const auto citAttr = data.attributes->find(ADDRESS_NAME_NAME_ATTR_NAME);
    REQUIRE(citAttr != data.attributes->end(), "Address name object does not contain address name attribute");
    return std::make_pair<maps::geolib3::Point2, TString>(std::move(objects[0].second), citAttr->second.c_str());
}

NYT::TNode createSchema() {
    return NYT::TNode::CreateList()
                .Add(NYT::TNode()("name", COLUMN_NAME_FEATURE_ID)("type", "int64"))
                .Add(NYT::TNode()("name", COLUMN_NAME_LAT)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_LON)("type", "double"))
                .Add(NYT::TNode()("name", COLUMN_NAME_OBJECTS)("type", "any"));
}

void extractAddressPoints(
    IClientPtr& client,
    const TString& resultYTTable,
    const maps::wiki::common::ExtendedXmlDoc& wikiConfig)
{
    maps::wiki::common::PoolHolder wiki(wikiConfig, "long-read", "long-read");
    auto txn = wiki.pool().slaveTransaction();
    maps::wiki::revision::RevisionsGateway gateway(*txn);
    const maps::wiki::revision::DBID commitId = gateway.headCommitId();
    INFO() << "Snapshot commit id " << commitId;

    maps::wiki::revision::Snapshot snapshot = gateway.snapshot(commitId);
    INFO() << "Created table: " << resultYTTable;
    TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(YT_TABLE);
    TTableWriterPtr<TNode> writer = client->CreateTableWriter<TNode>(resultYTTable);
    for (int processedItems = 0; reader->IsValid(); reader->Next(), processedItems++) {
        if ((processedItems + 1) % 1000000 == 0) {
            INFO() << "Processed " << (processedItems + 1) << " items";
        }

        const TNode& inpRow = reader->GetRow();
        if (inpRow[COL_TYPE].AsString() != FEEDBACK_TYPE ||
            inpRow[COL_SOURCE].AsString() != FEEDBACK_SOURCE ||
            inpRow[COL_RESOLUTION].IsNull() ||
            inpRow[COL_RESOLUTION] != FEEDBACK_ACCEPTED)
        {
            continue;
        }

        std::optional<std::pair<maps::geolib3::Point2, TString> >
            addrPoint = getAddressPoint(snapshot, extractIds(inpRow[COL_COMMIT_IDS]));
        if (!addrPoint) {
            continue;
        }

        TString addressHouseNumber = inpRow["attrs"]["addressHouseNumber"].AsString();
        const TVector<TNode>& imageFeatures = inpRow["attrs"]["sourceContext"]["content"]["imageFeatures"].AsList();
        for (const TNode& feature : imageFeatures) {
            if (TString::npos != feature["imageFull"]["url"].AsString().find("pano")) {
                continue;
            }
            const TVector<TNode>& featureBbox = feature["box"].AsList();
            const TVector<TNode>& featureCoordinates = feature["geometry"]["coordinates"].AsList();
            const TString featureId = feature["id"].AsString();

            TNode objectsNode;
            objectsNode(ITEM_NAME_ADDR_NUMBER, addrPoint->second)
                       (ITEM_NAME_ADDR_PT_LAT, addrPoint->first.y())
                       (ITEM_NAME_ADDR_PT_LON, addrPoint->first.x())
                       (ITEM_NAME_BBOX, TNode::CreateList()
                                            .Add(TNode::CreateList().Add(featureBbox[0]).Add(featureBbox[1]))
                                            .Add(TNode::CreateList().Add(featureBbox[2]).Add(featureBbox[3])))
                       (ITEM_NAME_HYP_NUM, addressHouseNumber);
            TNode outRow;
            outRow(COLUMN_NAME_FEATURE_ID, FromString<int64_t>(featureId))
                  (COLUMN_NAME_LAT, featureCoordinates[1].AsDouble())
                  (COLUMN_NAME_LON, featureCoordinates[0].AsDouble())
                  (COLUMN_NAME_OBJECTS, objectsNode);
            writer->AddRow(outRow);
        }
    };
    writer->Finish();
    client->Sort(
        NYT::TSortOperationSpec()
            .AddInput(resultYTTable)
            .Output(resultYTTable)
            .SortBy(COLUMN_NAME_FEATURE_ID)
    );
}

void saveAddressPoints(
    IClientPtr& client,
    const TString& addressPointYTTable,
    const TString& outputTableName,
    int64_t commitId = -1)
{
    client->Create(
        outputTableName.c_str(),
        NYT::NT_TABLE,
        NYT::TCreateOptions()
        .Recursive(true)
        .Attributes(
            NYT::TNode()
            ("schema", createSchema())
            (TABLE_ATTR_NAME_COMMIT_ID, commitId)
        )
    );

    TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(addressPointYTTable);
    TTableWriterPtr<TNode> writer = client->CreateTableWriter<TNode>(outputTableName);
    int64_t lastFeatureId = -1;
    TNode outRow;
    TNode objectsNode = TNode::CreateList();
    for (; reader->IsValid(); reader->Next()) {
        const TNode& inpRow = reader->GetRow();
        const int64_t featureId = inpRow[COLUMN_NAME_FEATURE_ID].AsInt64();
        if (lastFeatureId != featureId) {
            if (-1 != lastFeatureId) {
                outRow(COLUMN_NAME_OBJECTS, objectsNode);
                writer->AddRow(outRow);
            }
            lastFeatureId = featureId;
            outRow.Clear();
            outRow(COLUMN_NAME_FEATURE_ID, featureId)
                (COLUMN_NAME_LAT, inpRow[COLUMN_NAME_LAT].AsDouble())
                (COLUMN_NAME_LON, inpRow[COLUMN_NAME_LON].AsDouble());
            objectsNode.Clear();
            objectsNode.Add(inpRow[COLUMN_NAME_OBJECTS]);
        } else {
            objectsNode.Add(inpRow[COLUMN_NAME_OBJECTS]);
        }
    };
    writer->Finish();
}

class BuildingSearcher {
public:
    typedef maps::geolib3::StaticGeometrySearcher<maps::geolib3::Polygon2, int> InternalSearcher;
public:
    explicit BuildingSearcher(const maps::mrc::object::Buildings& buildings) {
        for (const maps::mrc::object::Building& building : buildings) {
            this->insert(building.geom());
        }
        this->build();
    }
    const InternalSearcher::SearchResult find(const maps::geolib3::BoundingBox& searchBox) const {
        return searcher_.find(searchBox);
    }
private:
    void insert(const maps::geolib3::Polygon2& pgn) {
        polygons_.push_back(pgn);
        searcher_.insert(&polygons_.back(), -1);
    }
    void build() {
        searcher_.build();
    }
private:
    InternalSearcher searcher_;
    std::list<maps::geolib3::Polygon2> polygons_;
};

struct ValidAddressPoint {
    ValidAddressPoint(
        const TString& hypNumber_,
        const TString& addrPtNumber_,
        const maps::geolib3::Point2& addrPtMercatorPos_)
        : hypNumber(hypNumber_.c_str())
        , addrPtNumber(addrPtNumber_.c_str())
        , addrPtMercatorPos(addrPtMercatorPos_)
    {}

    std::string hypNumber = "";
    std::string addrPtNumber = "";
    maps::geolib3::Point2 addrPtMercatorPos;
};

std::vector<ValidAddressPoint> loadValidAddressPoints(
    const TNode& objectsNode)
{
    std::vector<ValidAddressPoint> results;
    const TVector<TNode>& objectsList = objectsNode.AsList();
    for (const TNode& object : objectsList) {
        results.emplace_back(
            object[ITEM_NAME_HYP_NUM].AsString(),
            object[ITEM_NAME_ADDR_NUMBER].AsString(),
            maps::geolib3::convertGeodeticToMercator(
                maps::geolib3::Point2(
                    object[ITEM_NAME_ADDR_PT_LON].AsDouble(),
                    object[ITEM_NAME_ADDR_PT_LAT].AsDouble()
                )
            )
        );
    }
    return results;
}

struct DatasetRecord {
    std::string hypNumber;
    std::string candidateNumber;
    double distanceMeters;
    double perimeterMeters;
    bool positive;
};

std::ostream& operator<< (std::ostream& stream, const DatasetRecord& record) {
    stream  << record.hypNumber << "||"
            << record.candidateNumber << "||"
            << record.distanceMeters << "||"
            << record.perimeterMeters << "||"
            << (record.positive ? 1 : 0);
    return stream;
}

std::vector<DatasetRecord> getRecords(
    const TNode& inpRow,
    maps::mrc::object::Loader& loader)
{
    const maps::geolib3::Point2 featureMercatorPos =
        maps::geolib3::convertGeodeticToMercator(
            maps::geolib3::Point2(
                inpRow[COLUMN_NAME_LON].AsDouble(),
                inpRow[COLUMN_NAME_LAT].AsDouble()
            )
        );

    std::vector<ValidAddressPoint> validAddressPoints = loadValidAddressPoints(inpRow[COLUMN_NAME_OBJECTS]);

    const double mercatorSearchDiameter = 2. * maps::geolib3::toMercatorUnits(SEARCH_RADIUS_METERS, featureMercatorPos);
    const maps::geolib3::BoundingBox searchBbox(featureMercatorPos, mercatorSearchDiameter, mercatorSearchDiameter);
    maps::mrc::object::Buildings buildings = loader.loadBuildings(searchBbox);
    maps::geolib3::BoundingBox buildingsBBox = searchBbox;
    for (size_t i = 0; i < buildings.size(); i++) {
        const maps::mrc::object::Building& building = buildings[i];
        buildingsBBox = maps::geolib3::expand(buildingsBBox, building.geom());
    }

    BuildingSearcher buildingSearcher(buildings);
    const double mercatorEpsilon = maps::geolib3::toMercatorUnits(EPSILON_METERS, featureMercatorPos);
    const maps::mrc::object::AddressPointWithNames& addrPointsWithName = loader.loadAddressPointWithNames(buildingsBBox);
    std::vector<DatasetRecord> results;
    for (const maps::mrc::object::AddressPointWithName& addrPointWithName : addrPointsWithName) {
        bool buildingFound = false;
        const maps::geolib3::Point2& addrPoint = addrPointWithName.geom();
        auto buildingsForAddrPoint = buildingSearcher.find(addrPoint.boundingBox());
        maps::geolib3::Polygon2 buildingPolygon;
        for (auto itr = buildingsForAddrPoint.first; itr != buildingsForAddrPoint.second; itr++) {
            if (!maps::geolib3::spatialRelation(addrPoint, itr->geometry(), maps::geolib3::SpatialRelation::Within))
                continue;
            buildingPolygon = itr->geometry();
            buildingFound = true;
            break;
        }
        if (!buildingFound) {
            const double radius = maps::geolib3::toMercatorUnits(POINT_WITHOUT_BUILDING_RADIUS_METERS, addrPoint);
            buildingPolygon = maps::geolib3::BoundingBox(addrPoint, 2 * radius, 2 * radius).polygon();
        }
        for (size_t i = 0; i < validAddressPoints.size(); i++) {
            const ValidAddressPoint& validPoint = validAddressPoints[i];
            DatasetRecord record;
            record.hypNumber = validPoint.hypNumber;
            record.candidateNumber = addrPointWithName.name();
            record.distanceMeters = maps::geolib3::toMeters(maps::geolib3::distance(featureMercatorPos, buildingPolygon), featureMercatorPos);
            record.perimeterMeters = maps::geolib3::toMeters(buildingPolygon.perimeter(), featureMercatorPos);
            if (std::abs(validPoint.addrPtMercatorPos.x() - addrPoint.x()) < mercatorEpsilon &&
                std::abs(validPoint.addrPtMercatorPos.y() - addrPoint.y()) < mercatorEpsilon &&
                validPoint.addrPtNumber == addrPointWithName.name())
            {
                record.positive = true;
            } else {
                record.positive = false;
            }
            results.push_back(record);
        }
    }
    return results;
}

typedef TLockFreeQueue<NYT::TNode> NodeQueue;
typedef TLockFreeQueue<DatasetRecord> DatasetRecordQueue;

class DatasetRecordsExtractor
    : public IObjectInQueue {

public:
    DatasetRecordsExtractor(
        NodeQueue* inpQueue,
        DatasetRecordQueue* outQueue,
        const maps::wiki::common::ExtendedXmlDoc& wikiConfig)
        : inpQueue_(inpQueue)
        , outQueue_(outQueue)
        , wikiConfig_(wikiConfig)
        , running_(true) {
        AtomicSet(waitData_, 1);
    }
    void Process(void* /*threadSpecificResource*/) override {
        maps::wiki::common::PoolHolder wiki(wikiConfig_, "long-read", "long-read");
        auto txn = wiki.pool().slaveTransaction();
        maps::wiki::revision::RevisionsGateway gateway(*txn);
        const maps::wiki::revision::DBID commitId = gateway.headCommitId();
        INFO() << "Snapshot commit id " << commitId;
        maps::wiki::revision::Snapshot snapshot = gateway.snapshot(commitId);

        maps::mrc::object::LoaderHolder loader = maps::mrc::object::makeRevisionLoader(snapshot);
        for (;;) {
            NYT::TNode inpNode;
            if (inpQueue_->Dequeue(&inpNode)) {
                outQueue_->EnqueueAll(getRecords(inpNode, *loader));
            } else if (!isWaitData())
                break;
            else
                std::this_thread::sleep_for(THREAD_WAIT_TIMEOUT);
        }
        running_ = false;
    }
    void DataEnded() {
        AtomicSet(waitData_, 0);
    }
    bool isWaitData() {
        return (0 != AtomicGet(waitData_));
    }
    bool isRunning() const {
        return running_;
    }
private:
    NodeQueue* inpQueue_;
    DatasetRecordQueue* outQueue_;
    TAtomic waitData_;
    const maps::wiki::common::ExtendedXmlDoc& wikiConfig_;
    bool running_;
};

void extractDataset(
    IClientPtr& client,
    const TString& addressPointYTTable,
    const maps::wiki::common::ExtendedXmlDoc& wikiConfig,
    const std::string& outputDatasetPath,
    int threadsCount)
{
    if (threadsCount < 2) {
        maps::wiki::common::PoolHolder wiki(wikiConfig, "long-read", "long-read");
        auto txn = wiki.pool().slaveTransaction();
        maps::wiki::revision::RevisionsGateway gateway(*txn);
        const maps::wiki::revision::DBID commitId = gateway.headCommitId();
        INFO() << "Snapshot commit id " << commitId;
        maps::wiki::revision::Snapshot snapshot = gateway.snapshot(commitId);

        maps::mrc::object::LoaderHolder loader = maps::mrc::object::makeRevisionLoader(snapshot);

        TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(addressPointYTTable);
        std::ofstream ofs(outputDatasetPath.c_str());

        for (int processedItems = 0; reader->IsValid(); reader->Next(), processedItems++) {
            if ((processedItems + 1) % 1000 == 0) {
                INFO() << "Processed " << (processedItems + 1) << " items";
            }
            const TNode& inpRow = reader->GetRow();
            std::vector<DatasetRecord> records = getRecords(inpRow, *loader);
            for (size_t i = 0; i < records.size(); i++) {
                const DatasetRecord& record = records[i];
                ofs << record << std::endl;
            }
        }
    } else {
        TAutoPtr<IThreadPool> mtpQueue = CreateThreadPool(threadsCount);
        NodeQueue inpQueue;
        DatasetRecordQueue outQueue;
        std::vector<TAutoPtr<DatasetRecordsExtractor>> extractors;
        for (int i = 0; i < threadsCount; i++) {
            TAutoPtr<DatasetRecordsExtractor> extractor(new DatasetRecordsExtractor(
                &inpQueue,
                &outQueue,
                wikiConfig));
            mtpQueue->SafeAdd(extractor.Get());
            extractors.push_back(extractor);
        }

        TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(addressPointYTTable);
        for (int processedItems = 0; reader->IsValid(); reader->Next(), processedItems++) {
            if ((processedItems + 1) % 1000 == 0) {
                INFO() << "Processed " << (processedItems + 1) << " items";
            }
            inpQueue.Enqueue(reader->GetRow());
        }
        for (size_t i = 0; i < extractors.size(); i++) {
            extractors[i]->DataEnded();
        }
        std::ofstream ofs(outputDatasetPath.c_str());
        bool wait = true;
        for (;wait;) {
            wait = !inpQueue.IsEmpty();
            for (size_t i = 0; i < extractors.size(); i++) {
                wait |= extractors[i]->isRunning();
            }
            for (;;) {
                DatasetRecord record;
                if (!outQueue.Dequeue(&record))
                    break;
                ofs << record << std::endl;
            }
            std::this_thread::sleep_for(THREAD_WAIT_TIMEOUT);
        }
    }
}

}// namespace

int main(int argc, const char** argv) try {
    Initialize(argc, argv);

    maps::cmdline::Parser parser("Extract accepted hypotheses with commits from feedback");

    maps::cmdline::Option<std::string> wikiConfigPath = parser.string("wiki-config")
        .required()
        .help("Path to services config for wikimap");

    maps::cmdline::Option<std::string> outputTableName = parser.string("output")
        .help("Path to output YT table with address points");

    maps::cmdline::Option<std::string> inputTableName = parser.string("input")
        .help("Path to input YT table with address points");

    maps::cmdline::Option<std::string> outputDatasetPath = parser.string("output-ds")
        .help("Path to output dataset file");

    maps::cmdline::Option<int> threadsCount = parser.num("threads-cnt")
        .defaultValue(10);

    maps::cmdline::Option<bool> calculateStatistic = parser.flag("statistic")
        .help("Calculate statistic");


    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Connecting to yt::" << YT_PROXY;
    IClientPtr client = CreateClient(YT_PROXY);
    client->Create(YT_TEMPORARY_FOLDER, NYT::NT_MAP, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true));

    maps::wiki::common::ExtendedXmlDoc wikiConfig(wikiConfigPath);

    if (outputTableName.defined()) {
        const NYT::TTempTable tmpTable(client, "", YT_TEMPORARY_FOLDER);
        extractAddressPoints(client, tmpTable.Name(), wikiConfig);
        saveAddressPoints(client, tmpTable.Name(), outputTableName.c_str());
    }

    if (outputDatasetPath.defined()) {
        REQUIRE(inputTableName.defined(), "Input table not defined");
        extractDataset(
            client,
            inputTableName.c_str(),
            wikiConfig,
            outputDatasetPath,
            threadsCount
        );
    }

    if (calculateStatistic) {
        REQUIRE(inputTableName.defined(), "Input table not defined");
        int64_t sameNumbersCnt = 0;
        int64_t itemsCnt = 0;
        TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(inputTableName.c_str());
        std::vector<double> distances;
        for (; reader->IsValid(); reader->Next()) {
            const TNode& inpRow = reader->GetRow();
            const maps::geolib3::Point2 featurePos =
                maps::geolib3::Point2(
                    inpRow[COLUMN_NAME_LON].AsDouble(),
                    inpRow[COLUMN_NAME_LAT].AsDouble()
                );

            const TVector<TNode>& objectsList = inpRow[COLUMN_NAME_OBJECTS].AsList();
            for (const TNode& object : objectsList) {
                if (object[ITEM_NAME_ADDR_NUMBER].AsString() == object[ITEM_NAME_HYP_NUM].AsString()) {
                    sameNumbersCnt++;
                }
                itemsCnt++;

                const maps::geolib3::Point2 objPos =
                    maps::geolib3::Point2(
                        object[ITEM_NAME_ADDR_PT_LON].AsDouble(),
                        object[ITEM_NAME_ADDR_PT_LAT].AsDouble()
                    );
                const double distance = maps::geolib3::geoDistance(featurePos, objPos);
                if (distance > 150.) {
                    INFO() << inpRow[COLUMN_NAME_FEATURE_ID].AsInt64() << " distance: " << distance;
                    continue;
                }
                distances.push_back(distance);
            }
        };
        std::sort(distances.begin(), distances.end());
        INFO() << "Distance minimum:  " << distances.front();
        INFO() << "Distance maximum:  " << distances.back();
        INFO() << "Distance medium:   " << distances[distances.size() / 2];
        size_t n = distances.size();
        double average = 0.0;
        if (n != 0) {
            average = std::accumulate(distances.begin(), distances.end(), 0.0) / n;
        }
        INFO() << "Distance average:  " << average;
        double variance = 0.0;
        for (size_t i = 0; i < distances.size(); i++) {
            const double temp = distances[i] - average;
            variance += (temp * temp);
        }
        if (n != 0) {
            INFO() << "Distance variance: " << variance / n;
        }
        INFO() << "Item with same numbers: " << sameNumbersCnt << " from " << itemsCnt;
        return 0;
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Tools failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Tools failed: " << e.what();
    return EXIT_FAILURE;
}
