#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <geos/io/WKBReader.h>
#include <geos/geom/Point.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <list>
#include <map>


namespace {

void loadAddrCSVFile(const std::string &addrPath, const std::string &filterISO, std::map<uint64_t, uint64_t> &mapAddrIdToNodeId) {
    static const std::string COLUMN_NAME_ADDR_ID = "addr_id";
    static const std::string COLUMN_NAME_NODE_ID = "node_id";
    static const std::string COLUMN_NAME_ISOCODE = "isocode";

    bool filterISOEnable = !filterISO.empty();

    std::ifstream ifs(addrPath);
    REQUIRE(ifs.is_open(), "Unable to open addr table file");

    int columnIdxAddrID  = -1;
    int columnIdxNodeID  = -1;
    int columnIdxIsocode = -1;

    // found column index by name
    std::string line; std::getline(ifs, line);
    line += ',';
    const char *begin = line.c_str();
    int idx = 0;
    for (const char *ptr = begin; *ptr; ++ptr) {
        if (*ptr == ',') {
            if (ptr - begin == (int64_t)COLUMN_NAME_ADDR_ID.size() &&
                0 == strncmp(begin, COLUMN_NAME_ADDR_ID.c_str(), COLUMN_NAME_ADDR_ID.size())) {
                columnIdxAddrID = idx;
            } else if (ptr - begin == (int64_t)COLUMN_NAME_NODE_ID.size() &&
                0 == strncmp(begin, COLUMN_NAME_NODE_ID.c_str(), COLUMN_NAME_NODE_ID.size())) {
                columnIdxNodeID = idx;
            } else if (filterISOEnable &&
                ptr - begin == (int64_t)COLUMN_NAME_ISOCODE.size() &&
                0 == strncmp(begin, COLUMN_NAME_ISOCODE.c_str(), COLUMN_NAME_ISOCODE.size())) {
                columnIdxIsocode = idx;
            }
            begin = ptr + 1;
            idx++;
        }
    }

    REQUIRE(-1 != columnIdxAddrID, "Column with addr_id not found in addr csv file table");
    REQUIRE(-1 != columnIdxNodeID, "Column with node_id not found in addr csv file table");
    REQUIRE((-1 != columnIdxIsocode) || (!filterISOEnable),
            "Column with iso code not found in addr csv file table");

    int needReadColumns = filterISOEnable ? 3 : 2;
    for (; !ifs.eof();) {
        std::getline(ifs, line);
        if (line.empty())
            continue;

        int64_t addrID = -1;
        int64_t nodeID = -1;
        begin = line.c_str();
        idx = 0;
        int readedColumns = 0;
        const char *ptr = begin;
        for (; *ptr; ++ptr) {
            if (*ptr == ',') {
                if (idx == columnIdxAddrID) {
                    addrID = atoll(begin);
                    readedColumns++;
                    if (readedColumns == needReadColumns)
                        break;
                } else if (idx == columnIdxNodeID) {
                    nodeID = atoll(begin);
                    readedColumns++;
                    if (readedColumns == needReadColumns)
                        break;
                } else if (filterISOEnable && idx == columnIdxIsocode) {
                    if (ptr - begin != (int64_t)filterISO.size() || 0 != strncmp(begin, filterISO.c_str(), filterISO.size())) {
                        addrID = -1;
                        break;
                    }
                    readedColumns++;
                    if (readedColumns == needReadColumns)
                        break;
                }
                begin = ptr + 1;
                idx++;
            }
        }
        if (readedColumns < needReadColumns) {
            if (idx == columnIdxAddrID)
                addrID = atoll(begin);
            else if (idx == columnIdxNodeID)
                nodeID = atoll(begin);
            else if (filterISOEnable && idx == columnIdxIsocode) {
                if (ptr - begin != (int64_t)filterISO.size() || 0 != strncmp(begin, filterISO.c_str(), filterISO.size())) {
                    addrID = -1;
                }
            }
        }
        if (addrID != -1 && nodeID != -1) {
            mapAddrIdToNodeId[addrID] = nodeID;
        }
    }
}

void mapNodeIDToAddressName(const std::string &addrNamePath, const std::map<uint64_t, uint64_t> &mapAddrIdToNodeId, std::map<uint64_t, std::string> &mapNodeIdToName) {
    static const std::string COLUMN_NAME_ADDR_ID = "addr_id";
    static const std::string COLUMN_NAME_NAME    = "name";

    std::ifstream ifs(addrNamePath);
    REQUIRE(ifs.is_open(), "Unable to open addr_nm table file");

    int columnIdxAddrID = -1;
    int columnIdxName   = -1;

    // found column index by name
    std::string line; std::getline(ifs, line);
    line += ',';
    const char *begin = line.c_str();
    int idx = 0;
    for (const char *ptr = begin; *ptr; ++ptr) {
        if (*ptr == ',') {
            if (ptr - begin == (int64_t)COLUMN_NAME_ADDR_ID.size() &&
                0 == strncmp(begin, COLUMN_NAME_ADDR_ID.c_str(), COLUMN_NAME_ADDR_ID.size())) {
                columnIdxAddrID = idx;
            } else if (ptr - begin == (int64_t)COLUMN_NAME_NAME.size() &&
                0 == strncmp(begin, COLUMN_NAME_NAME.c_str(), COLUMN_NAME_NAME.size())) {
                columnIdxName = idx;
            }
            begin = ptr + 1;
            idx++;
        }
    }

    REQUIRE(-1 != columnIdxAddrID, "Column with addr_id not found in addr_nm csv file table");
    REQUIRE(-1 != columnIdxName,   "Column with name not found in addr_nm csv file table");

    int needReadColumns = 2;
    for (; !ifs.eof();) {
        std::getline(ifs, line);
        if (line.empty())
            continue;

        int64_t addrID = -1;
        std::string name;
        begin = line.c_str();
        idx = 0;
        int readedColumns = 0;
        const char *ptr = begin;
        for (; *ptr; ++ptr) {
            if (*ptr == ',') {
                if (idx == columnIdxAddrID) {
                    addrID = atoll(begin);
                    readedColumns++;
                    if (readedColumns == needReadColumns)
                        break;
                } else if (idx == columnIdxName) {
                    name = std::string(begin, ptr - begin);
                    readedColumns++;
                    if (readedColumns == needReadColumns)
                        break;
                }
                begin = ptr + 1;
                idx++;
            }
        }
        if (readedColumns < needReadColumns) {
            if (idx == columnIdxAddrID)
                addrID = atoll(begin);
            else if (idx == columnIdxName)
                name = std::string(begin, ptr - begin);
        }

        if (addrID != -1 && !name.empty()) {
            std::map<uint64_t, uint64_t>::const_iterator cit = mapAddrIdToNodeId.find(addrID);
            if (cit != mapAddrIdToNodeId.end()) {
                mapNodeIdToName.insert(std::pair<uint64_t, std::string>(cit->second, name));
            }
        }
    }
}

class PointsSearcher {
public:
    typedef maps::geolib3::StaticGeometrySearcher<maps::geolib3::Point2, std::string> InternalSearcher;
public:
    void insert(maps::geolib3::Point2 &&pt, const std::string &name) {
        pts_.emplace_back(pt);
        searcher_.insert(&pts_.back(), name);
    }
    void build() {
        searcher_.build();
    }
    const InternalSearcher::SearchResult find(const maps::geolib3::BoundingBox& searchBox) const {
        return searcher_.find(searchBox);
    }
private:
    InternalSearcher searcher_;
    std::list<maps::geolib3::Point2> pts_;
};

void fillGeometrySearcher(const std::string &nodePath,
                          const std::map<uint64_t, std::string> &mapNodeIdToName,
                          PointsSearcher &searcher) {
    static const std::string COLUMN_NAME_NODE_ID = "node_id";
    static const std::string COLUMN_NAME_SHAPE   = "shape";

    std::ifstream ifs(nodePath);
    REQUIRE(ifs.is_open(), "Unable to open node table file");

    int columnIdxNodeID = -1;
    int columnIdxShape  = -1;

    // found column index by name
    std::string line; std::getline(ifs, line);
    line += ',';
    const char *begin = line.c_str();
    int idx = 0;
    for (const char *ptr = begin; *ptr; ++ptr) {
        if (*ptr == ',') {
            if (ptr - begin == (int64_t)COLUMN_NAME_NODE_ID.size() &&
                0 == strncmp(begin, COLUMN_NAME_NODE_ID.c_str(), COLUMN_NAME_NODE_ID.size())) {
                columnIdxNodeID = idx;
            } else if (ptr - begin == (int64_t)COLUMN_NAME_SHAPE.size() &&
                0 == strncmp(begin, COLUMN_NAME_SHAPE.c_str(), COLUMN_NAME_SHAPE.size())) {
                columnIdxShape = idx;
            }
            begin = ptr + 1;
            idx++;
        }
    }

    REQUIRE(-1 != columnIdxNodeID, "Column with node_id not found in node csv file table");
    REQUIRE(-1 != columnIdxShape,  "Column with shape not found in node csv file table");

    int needReadColumns = 2;

    for (; !ifs.eof();) {
        std::getline(ifs, line);
        if (line.empty())
            continue;

        int64_t nodeID = -1;
        std::string shape;
        begin = line.c_str();
        idx = 0;
        int readedColumns = 0;
        const char *ptr = begin;
        for (; *ptr; ++ptr) {
            if (*ptr == ',') {
                if (idx == columnIdxNodeID) {
                    nodeID = atoll(begin);
                    readedColumns++;
                    if (readedColumns == needReadColumns)
                        break;
                } else if (idx == columnIdxShape) {
                    shape = std::string(begin, ptr - begin);
                    readedColumns++;
                    if (readedColumns == needReadColumns)
                        break;
                }
                begin = ptr + 1;
                idx++;
            }
        }
        if (readedColumns < needReadColumns) {
            if (idx == columnIdxNodeID)
                nodeID = atoll(begin);
            else if (idx == columnIdxShape)
                shape = std::string(begin, ptr - begin);
        }

        if (nodeID != -1 && !shape.empty()) {
            std::map<uint64_t, std::string>::const_iterator cit = mapNodeIdToName.find(nodeID);
            if (cit != mapNodeIdToName.end()) {
                std::stringstream ss(shape);
                std::unique_ptr<geos::geom::Geometry> geom(geos::io::WKBReader().readHEX(ss));
                REQUIRE(dynamic_cast<geos::geom::Point*>(geom.get()), "Incorrect geometry type");
                geos::geom::Point *pt = dynamic_cast<geos::geom::Point*>(geom.get());
                searcher.insert({pt->getX(), pt->getY()}, cit->second);
            }
        }
    }
}

struct HouseSignNumber {
    int boxXMin, boxYMin, boxXMax, boxYMax;
    std::string type;
    std::string num;
};

struct FeatureData {
    int64_t feature_id;
    std::string source;
    int orientation;
    double lon;
    double lat;
    int imgWidth;
    int imgHeight;
    std::string image;
    std::vector<HouseSignNumber> objects;
};

static const std::string JSON_NAME_FEATURE_ID  = "feature_id";
static const std::string JSON_NAME_SOURCE      = "source";
static const std::string JSON_NAME_ORIENTATION = "orientation";
static const std::string JSON_NAME_LON         = "lon";
static const std::string JSON_NAME_LAT         = "lat";
static const std::string JSON_NAME_IMG_WIDTH   = "img_width";
static const std::string JSON_NAME_IMG_HEIGHT  = "img_height";
static const std::string JSON_NAME_HEADING     = "heading";
static const std::string JSON_NAME_IMAGE       = "image";
static const std::string JSON_NAME_OBJECTS     = "objects";

static const std::string JSON_NAME_BBOX        = "bbox";
static const std::string JSON_NAME_TYPE        = "type";
static const std::string JSON_NAME_NUM         = "num";

void parseJson(const std::string &line, FeatureData &feature, double posShift) {
    const double shiftLatitude = posShift / maps::geolib3::WGS84_MAJOR_SEMIAXIS * 180. / M_PI;

    maps::json::Value jsonItem = maps::json::Value::fromString(line);
    feature.feature_id  = jsonItem[JSON_NAME_FEATURE_ID].as<int64_t>();
    feature.source      = jsonItem[JSON_NAME_SOURCE].as<std::string>();
    feature.orientation = jsonItem[JSON_NAME_ORIENTATION].as<int>();
    feature.lon         = jsonItem[JSON_NAME_LON].as<double>();
    feature.lat         = jsonItem[JSON_NAME_LAT].as<double>();
    if (jsonItem.hasField(JSON_NAME_IMAGE))
        feature.image = jsonItem[JSON_NAME_IMAGE].as<std::string>();
    if (jsonItem.hasField(JSON_NAME_HEADING)) {
        const double heading = jsonItem[JSON_NAME_HEADING].as<double>();
        feature.lat += shiftLatitude * cos(heading * M_PI / 180.);
        feature.lon += shiftLatitude * sin(heading * M_PI / 180.) / cos(feature.lat * M_PI / 180.);
    }
    if (jsonItem.hasField(JSON_NAME_IMG_WIDTH))
        feature.imgWidth = jsonItem[JSON_NAME_IMG_WIDTH].as<int>();
    if (jsonItem.hasField(JSON_NAME_IMG_HEIGHT))
        feature.imgHeight = jsonItem[JSON_NAME_IMG_HEIGHT].as<int>();
    maps::json::Value objects = jsonItem[JSON_NAME_OBJECTS];
    feature.objects.resize(objects.size());
    for (size_t i = 0; i < objects.size(); i++) {
        feature.objects[i].boxXMin = objects[i][JSON_NAME_BBOX][0][0].as<int>();
        feature.objects[i].boxYMin = objects[i][JSON_NAME_BBOX][0][1].as<int>();
        feature.objects[i].boxXMax = objects[i][JSON_NAME_BBOX][1][0].as<int>();
        feature.objects[i].boxYMax = objects[i][JSON_NAME_BBOX][1][1].as<int>();
        feature.objects[i].type = objects[i][JSON_NAME_TYPE].as<std::string>();
        feature.objects[i].num  = objects[i][JSON_NAME_NUM].as<std::string>();
    }
}

std::string saveJson(const HouseSignNumber &object) {
    std::stringstream ss;
    ss << "{"
       << "\"" << JSON_NAME_BBOX << "\": "
       << "["
       << "["  << object.boxXMin << ", "     << object.boxYMin << "], "
       << "["  << object.boxXMax << ", "     << object.boxYMax << "] "
       << "],"
       << "\"" << JSON_NAME_TYPE << "\": \"" << object.type << "\", "
       << "\"" << JSON_NAME_NUM  << "\": \"" << object.num  << "\""
       << "}";
    return ss.str();
}

std::string saveJson(const FeatureData &feature) {
    std::stringstream ss;
    ss << "{"
       << "\"" << JSON_NAME_FEATURE_ID  << "\": "   << feature.feature_id  << ", "
       << "\"" << JSON_NAME_SOURCE      << "\": \"" << feature.source      << "\", "
       << "\"" << JSON_NAME_ORIENTATION << "\": "   << feature.orientation << ", "
       << "\"" << JSON_NAME_LON         << "\": "   << feature.lon         << ", "
       << "\"" << JSON_NAME_LAT         << "\": "   << feature.lat         << ", "
       << "\"" << JSON_NAME_IMG_WIDTH   << "\": "   << feature.imgWidth    << ", "
       << "\"" << JSON_NAME_IMG_HEIGHT  << "\": "   << feature.imgHeight   << ", ";
       if (!feature.image.empty())
           ss << "\"" << JSON_NAME_IMAGE       << "\": \"" << feature.image       << "\", ";
       ss << "\"" << JSON_NAME_OBJECTS     << "\": [";
    ss << saveJson(feature.objects[0]);
    for (size_t i = 1; i < feature.objects.size(); i++) {
        ss << ", " << saveJson(feature.objects[i]);
    }
    ss << "]" << "}";
    return ss.str();
}

int ToNumber(const std::string &str) {
    int result = 0;
    const char *p = str.c_str();
    for (; *p; p++) {
        if (!std::isdigit(*p))
            continue;
        result = result * 10 + (*p - '0');
    }
    return result;
}

void filterJson(const std::string &inputJsonPath,
                const std::string &outputJsonPath,
                const std::string &goodJsonPath,
                const PointsSearcher &searcher,
                double searchRadius, double shiftDistance) {
    std::ifstream ifs(inputJsonPath);
    REQUIRE(ifs.is_open(), "Unable to open input json file");

    std::ofstream ofs(outputJsonPath);
    REQUIRE(ofs.is_open(), "Unable to open ouput json file");

    std::ofstream ofsGood;
    const bool saveGood = (!goodJsonPath.empty());
    if (saveGood) {
        ofsGood.open(goodJsonPath);
        REQUIRE(ofsGood.is_open(), "Unable to open json file for numbers found in ymasdf");
    }

    FeatureData feature;
    const double searchDiameterLatitude = 2.0 * searchRadius / maps::geolib3::WGS84_MAJOR_SEMIAXIS
                                              * 180. / M_PI;
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        parseJson(line, feature, shiftDistance);
        INFO() << "Feature: " << feature.feature_id
               << ", location: " << feature.lon << ", " << feature.lat
               << ", signs count: " << feature.objects.size();

        maps::geolib3::BoundingBox bbox(maps::geolib3::Point2(feature.lon, feature.lat),
                                        searchDiameterLatitude / cos(feature.lat * M_PI / 180.),
                                        searchDiameterLatitude);

        FeatureData featureGood = feature;
        featureGood.objects.clear();
        auto searchResult = searcher.find(bbox);
        for (auto itr = searchResult.first; itr != searchResult.second; itr++) {
            for (auto itn = feature.objects.begin(); itn != feature.objects.end();) {
                if (ToNumber(itn->num) == ToNumber(itr->value())) {
                    featureGood.objects.push_back((*itn));
                    itn = feature.objects.erase(itn);
                } else {
                    itn++;
                }
            }
        }
        if (0 != feature.objects.size()) {
            ofs << saveJson(feature) << std::endl;
        }
        if (saveGood && 0 != featureGood.objects.size()) {
            ofsGood << saveJson(featureGood) << std::endl;
        }
    }
}

} //namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Load addrespoint information from ymasdf csv");

    maps::cmdline::Option<std::string> addrNamePath = parser.string("addr_nm")
        .required()
        .help("Path to addr_nm table in csv format");

    maps::cmdline::Option<std::string> addrPath = parser.string("addr")
        .required()
        .help("Path to addr table in csv format");

    maps::cmdline::Option<std::string> nodePath = parser.string("node")
        .required()
        .help("Path to node table in csv format");

    maps::cmdline::Option<std::string> isocode = parser.string("isocode")
        .defaultValue("")
        .help("isocode for filter in addr table. If blank - no filter apply.");

    maps::cmdline::Option<std::string> inputJsonPath = parser.string("input_json")
        .required()
        .help("Path to input json file for filter");

    maps::cmdline::Option<std::string> outputJsonPath = parser.string("output_json")
        .required()
        .help("Path to output json file");

    maps::cmdline::Option<std::string> goodJsonPath = parser.string("good_json")
        .defaultValue("")
        .help("Path to json file for save (optionaly) data with numbers found on ymapsdf");

    maps::cmdline::Option<double> searchRadius = parser.real("radius")
        .defaultValue(50.)
        .help("Radius of geometric search in meters. (default: 50)");

    maps::cmdline::Option<double> shiftDistance = parser.real("shift")
        .defaultValue(25.)
        .help("Shift image coordinates in heading direction. (default: 25)");


    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Load addr_id to node_id map";
    std::map<uint64_t, uint64_t> mapAddrIdToNodeId;
    loadAddrCSVFile(addrPath, isocode, mapAddrIdToNodeId);

    INFO() << "Load node_id to name map";
    std::map<uint64_t, std::string> mapNodeIdToName;
    mapNodeIDToAddressName(addrNamePath, mapAddrIdToNodeId, mapNodeIdToName);

    INFO() << "Fill geometry searcher";
    PointsSearcher searcher;
    fillGeometrySearcher(nodePath, mapNodeIdToName, searcher);
    searcher.build();

    INFO() << "Start filter json file";
    filterJson(inputJsonPath, outputJsonPath, goodJsonPath, searcher, searchRadius, shiftDistance);

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
