#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/std.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <mapreduce/yt/interface/client.h>

#include <random>
#include <string>
#include <list>
#include <fstream>

using namespace NYT;

void SaveBBox(const TNode& node, maps::json::ObjectBuilder &builder) {
    builder["bbox"] << [&node](maps::json::ArrayBuilder builder) {
        const TVector<TNode>& rcNode = node.AsList();
        builder << [&rcNode](maps::json::ArrayBuilder builder) {
                builder << rcNode[0].AsList()[0].AsInt64();
                builder << rcNode[0].AsList()[1].AsInt64();
            };
        builder << [&rcNode](maps::json::ArrayBuilder builder) {
                builder << rcNode[1].AsList()[0].AsInt64();
                builder << rcNode[1].AsList()[1].AsInt64();
            };
    };
}

void SaveObjects(const TNode& node, maps::json::ObjectBuilder &builder) {
    const TVector<TNode>& objectList = node.AsList();
    builder["objects"] << [&](maps::json::ArrayBuilder builder) {
        for (const TNode& objectNode : objectList) {
            builder << [&objectNode](maps::json::ObjectBuilder builder) {
                builder["type"] = objectNode["type"].AsString();
                builder["num"]  = objectNode["num"].AsString();;
                SaveBBox(objectNode["bbox"], builder);
            };
        };
    };
}

void Download(IClientPtr client,
              const NYT::TRichYPath &inputTable,
              const TString &outputPath) {
    INFO() << "Make result json";

    std::ofstream ofs(outputPath);
    if (!ofs.is_open())
        ERROR() << "Unable to open output file: " << outputPath;
    maps::json::Builder builder(ofs);

    builder << [&](maps::json::ObjectBuilder builder) {
        builder["images_with_objects"] << [&](maps::json::ArrayBuilder builder) {
            TTableReaderPtr<TNode> reader = client->CreateTableReader<TNode>(inputTable);
            for (int processedItems = 0; reader->IsValid(); reader->Next(), processedItems++) {
                if ((processedItems + 1) % 1000 == 0) {
                    INFO() << "Processed " << (processedItems + 1) << " items";
                }
                const TNode& inpRow = reader->GetRow();
                builder << [&](maps::json::ObjectBuilder builder) {
                    builder["feature_id"] << inpRow["feature_id"].AsInt64();
                    builder["lon"] << inpRow["lon"].AsDouble();
                    builder["lat"] << inpRow["lat"].AsDouble();
                    builder["image"] << inpRow["image"].AsString();
                    SaveObjects(inpRow["objects"], builder);
                };
            };
        };
    };
}

int main(int argc, const char** argv) try {
    Initialize(argc, argv);

    maps::cmdline::Parser parser("Download house number dataset from YT table to json");

    maps::cmdline::Option<std::string>
        inputTablePath = parser.string("input")
                               .required()
                               .help("Input YT table path");

    maps::cmdline::Option<std::string>
        outputJsonPath = parser.string("output")
                               .required()
                               .help("Path to output json file");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Connecting to yt::hahn";
    IClientPtr client = CreateClient("hahn");
    Download(client, inputTablePath.c_str(), outputJsonPath.c_str());
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
