#include <maps/wikimap/mapspro/services/mrc/libs/house_number_sign_detector/include/house_number_sign_detector.h>
#include <maps/wikimap/mapspro/services/mrc/libs/carsegm/include/carsegm.h>

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/http/include/http.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/value.h>

#include <opencv2/opencv.hpp>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <chrono>

using namespace std::chrono;
using namespace maps::mrc::house_number_sign_detector;
using namespace maps::mrc::carsegm;

namespace {

struct Feature {
    int64_t featureID;
    std::string url;
    int orientation;
    double x;
    double y;
    cv::Size imgSize;
};

void applyOrientation(cv::Mat &image, int orientation){
    switch (orientation) {
    case 2:
        cv::flip(image, image, 1);
        break;
    case 3:
        cv::flip(image, image, -1);
        break;
    case 4:
        cv::flip(image, image, 0);
        break;
    case 5:
        image = image.t();
        break;
    case 6:
        image = image.t();
        cv::flip(image, image, 1);
        break;
    case 7:
        image = image.t();
        cv::flip(image, image, -1);
        break;
    case 8:
        image = image.t();
        cv::flip(image, image, 0);
        break;
    }
}

void revertBoxOrientation(cv::Rect &box, int orientation, const cv::Size &imageSize) {
    if (box.width < 0) {
        box.x += (box.width + 1);
        box.width = -box.width;
    }
    if (box.height < 0) {
        box.y += (box.height + 1);
        box.height = -box.height;
    }

    switch (orientation) {
    case 2:
        box.x = imageSize.width - 1 - (box.x + box.width - 1);
        return ;
    case 3:
        box.x = imageSize.width  - 1 - (box.x + box.width - 1);
        box.y = imageSize.height - 1 - (box.y + box.height - 1);
        break;
    case 4:
        box.y = imageSize.height - 1 - (box.y + box.height - 1);
        break;
    case 5:
        std::swap(box.x, box.y);
        std::swap(box.width, box.height);
        break;
    case 6:
        box.x = imageSize.width - 1 - (box.x + box.width - 1);
        std::swap(box.x, box.y);
        std::swap(box.width, box.height);
        break;
    case 7:
        box.x = imageSize.width  - 1 - (box.x + box.width - 1);
        box.y = imageSize.height - 1 - (box.y + box.height - 1);
        std::swap(box.x, box.y);
        std::swap(box.width, box.height);
        break;
    case 8:
        box.y = imageSize.height - 1 - (box.y + box.height - 1);
        std::swap(box.x, box.y);
        std::swap(box.width, box.height);
        break;
    }
}

std::set<int64_t> loadFilterFeatureIDs(const std::string &filterPath) {
    std::set<int64_t> filterFeatureIDs;
    std::ifstream ifs(filterPath);
    if (!ifs.is_open())
        return filterFeatureIDs;
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        filterFeatureIDs.insert(std::stoll(line));
    }
    return filterFeatureIDs;
}

std::string download(maps::http::Client& client, const std::string& url, size_t retryCnt) {
    maps::common::RetryPolicy retryPolicy;
    retryPolicy.setTryNumber(retryCnt);

    auto validateResponse = [](const auto& maybeResponse) {
        return maybeResponse.valid() && maybeResponse.get().responseClass() != maps::http::ResponseClass::ServerError;
    };
    auto resp = maps::common::retry(
                [&]() {
                    return maps::http::Request(client, maps::http::GET, maps::http::URL(url)).perform();
                },
                retryPolicy,
                validateResponse
            );
    if (resp.responseClass() != maps::http::ResponseClass::Success) {
        INFO() << "Unexpected response status " << resp.status() << " for url " << url;
        return "";
    }
    return resp.readBody();
}

void filterHouseNumberOnCars(
    const cv::Mat& image,
    const CarSegmentator& carSegmentator,
    HouseNumberSigns& signs)
{
    if (0 == signs.size())
        return;
    cv::Mat mask = carSegmentator.segment(image);
    signs.erase(
        std::remove_if(signs.begin(), signs.end(),
            [&](const HouseNumberSign& houseNumber) {
                return 2 * cv::countNonZero(mask(houseNumber.box)) >
                            houseNumber.box.area();
            }
        ),
        signs.end()
    );
}

void saveBBox(const cv::Rect& bbox, maps::json::ObjectBuilder &builder) {
    builder["bbox"] << [&bbox](maps::json::ArrayBuilder builder) {
        builder << [&bbox](maps::json::ArrayBuilder builder) {
                builder << bbox.x;
                builder << bbox.y;
            };
        builder << [&bbox](maps::json::ArrayBuilder builder) {
                builder << bbox.x + bbox.width;
                builder << bbox.y + bbox.height;
            };
    };
}

void saveObjects(const HouseNumberSigns& objects, maps::json::ObjectBuilder &builder) {
    static const std::string HOUSE_NUMBER_SIGN_TYPE_NAME = "house_number_sign";

    builder["objects"] << [&objects](maps::json::ArrayBuilder builder) {
        for (size_t i = 0; i < objects.size(); i++) {
                const HouseNumberSign &object = objects[i];
                builder << [&object](maps::json::ObjectBuilder builder) {
                builder["type"] = HOUSE_NUMBER_SIGN_TYPE_NAME;
                builder["num"]  = object.number;
                saveBBox(object.box, builder);
            };
        };
    };
}

void saveFeatureWithObjects(const Feature &feature, const HouseNumberSigns &objects, maps::json::Builder &builder) {
    builder << [&feature, &objects](maps::json::ObjectBuilder builder) {
        builder["feature_id"] << feature.featureID;
        builder["source"]     << feature.url;
        builder["orientation"]<< feature.orientation;
        builder["lon"]<< feature.x;
        builder["lat"]<< feature.y;
        builder["img_width"] << feature.imgSize.width;
        builder["img_height"] << feature.imgSize.height;
        saveObjects(objects, builder);
    };
}

void saveObjectOfFeature(const Feature &feature, const HouseNumberSign &object, maps::json::Builder &builder) {
    builder << [&feature, &object](maps::json::ObjectBuilder builder) {
        builder["id"] << feature.featureID;
        //maps::geolib3::Point2 pt = maps::geolib3::convertGeodeticToMercator(maps::geolib3::Point2(feature.lon, feature.lat));
        //builder["x"]<< pt.x();
        //builder["y"]<< pt.y();
        builder["x"]<< feature.x;
        builder["y"]<< feature.y;
        builder["number"] << object.number;
        builder["bbox"] << [&](maps::json::ArrayBuilder builder) {
            builder << object.box.x;
            builder << object.box.y;
            builder << object.box.x + object.box.width - 1;
            builder << object.box.y + object.box.height - 1;
        };
    };
}

} //namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Test time of classifiers. Classifier of image rotation enable always");

    maps::cmdline::Option<std::string> inputPath = parser.string("input")
        .required()
        .help("Path to list of images in text file with one image on the line:\
               <feature_id> <mds_url> <orientation>");

    maps::cmdline::Option<std::string> outputPath = parser.string("output")
        .required()
        .help("Path to file for save results. Results for one image on the line in json format");

    maps::cmdline::Option<std::string> filterPath = parser.string("filter")
        .defaultValue("")
        .help("Path to text file with feature ids witch already checked");

    maps::cmdline::Option<size_t> fromIndex = parser.size_t("from_idx")
        .defaultValue(0)
        .help("Index of start line in input file");

    maps::cmdline::Option<size_t> toIndex = parser.size_t("to_idx")
        .defaultValue(-1)
        .help("Index of line after last processed in input file");

    maps::cmdline::Option<size_t> retryCnt = parser.size_t("retry_cnt")
        .defaultValue(20)
        .help("Retry for download image");

    maps::cmdline::Option<bool> recognizeNum = parser.flag("recognize_num")
        .help("Recognize numbers");

    maps::cmdline::Option<bool> filterCar = parser.flag("filter_car")
        .help("Find car on image and remove house numbers on car");

    maps::cmdline::Option<bool> signdetectorFormat = parser.flag("signdetector_format")
        .help("Save results in format ready for upload to YT table and use in signdetector functions");

    parser.parse(argc, const_cast<char**>(argv));

    std::set<int64_t> filterFeatureIDs = loadFilterFeatureIDs(filterPath);

    std::ifstream ifs(inputPath);
    std::string line;
    size_t inpLineIdx = 0;
    for (; inpLineIdx < fromIndex && !ifs.eof(); inpLineIdx++) {
        std::getline(ifs, line);
    }

    INFO() << inpLineIdx << " skipped";

    maps::http::Client client;
    std::ofstream reworked(filterPath, std::ofstream::out | std::ofstream::app);
    std::ofstream ofs(outputPath, std::ofstream::out | std::ofstream::app);
    Feature feature;
    FasterRCNNDetector detector;
    CarSegmentator carSegmentator;
    for (; inpLineIdx < toIndex && !ifs.eof(); inpLineIdx++) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        std::stringstream ss(line);
        ss >> feature.featureID
           >> feature.url
           >> feature.orientation
           >> feature.x
           >> feature.y;
        if (filterFeatureIDs.find(feature.featureID) != filterFeatureIDs.end()) {
            INFO() << feature.featureID << " skipped (as already processed)";
            continue;
        }
        std::string temp = download(client, feature.url, retryCnt);
        if (temp.empty()) {
            INFO() << feature.featureID << " skipped (no data downloaded by url)";
            continue;
        }
        cv::Mat encImage((int)temp.size(), 1, CV_8UC1, (void *)temp.c_str());
        cv::Mat image;
        try {
            image = cv::imdecode(encImage, cv::IMREAD_COLOR | cv::IMREAD_IGNORE_ORIENTATION);
        } catch (...){
            INFO() << feature.featureID << " skipped (unable to decode image from url: " << feature.url << ")";
            continue;
        }
        if (image.empty()) {
            INFO() << feature.featureID << " skipped (unable to decode image from url: " << feature.url << ")";
            continue;
        }

        feature.imgSize = image.size();
        applyOrientation(image, feature.orientation);
        HouseNumberSigns signs = detector.detect(image,
                                                 recognizeNum ? RecognizeNumber::Yes : RecognizeNumber::No);
        if (filterCar)
            filterHouseNumberOnCars(image, carSegmentator, signs);
        if (0 < signs.size()) {
            for (size_t i = 0; i < signs.size(); i++) {
                revertBoxOrientation(signs[i].box, feature.orientation, image.size());
            }
            if (signdetectorFormat) {
                for (size_t i = 0; i < signs.size(); i++) {
                    std::stringstream jsonLine;
                    maps::json::Builder builder(jsonLine);
                    saveObjectOfFeature(feature, signs[i], builder);
                    ofs << jsonLine.str() << std::endl; ofs.flush();
                }
            } else {
                std::stringstream jsonLine;
                maps::json::Builder builder(jsonLine);
                saveFeatureWithObjects(feature, signs, builder);
                ofs << jsonLine.str() << std::endl; ofs.flush();
            }
        }
        reworked << feature.featureID << std::endl; reworked.flush();
        INFO() << feature.featureID << " processed, found " << signs.size() << " objects";
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
