#include <maps/wikimap/mapspro/services/mrc/libs/house_number_sign_detector/include/house_number_sign_detector.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/base64.h>
#include <maps/libs/http/include/http.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/value.h>

#include <library/cpp/deprecated/atomic/atomic.h>

#include <util/thread/pool.h>
#include <util/thread/lfqueue.h>

#include <opencv2/opencv.hpp>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <chrono>

using namespace std::chrono;
using namespace maps::mrc::house_number_sign_detector;

namespace {

constexpr std::chrono::milliseconds THREAD_WAIT_TIMEOUT(1);

struct CutParams {
    cv::Size tileSize;
    std::vector<cv::Size> photoSizes; // size of photo indexed by zoom levels
};

typedef std::map<int64_t, CutParams> MapCutParams;

struct PanoramasFeature {
    double lon;
    double lat;
    std::string mdsKey;
    int64_t cutParamsID;
};

struct ClusterOfSigns {
    cv::Rect rc; // rectangle on the image in the image coordinates
    HouseNumberSigns signs; // signs which in rc. Coordinates of box of signs in coordinate system with origin in top left of rc.
};

MapCutParams loadCutParams(const maps::json::Value &jsonCutParams) {
    static const std::string JSON_NAME_TILE   = "tile";
    static const std::string JSON_NAME_LEVEL  = "level";
    static const std::string JSON_NAME_WIDTH  = "width";
    static const std::string JSON_NAME_HEIGHT = "height";
    static const std::string JSON_NAME_ZOOMS  = "zooms";

    std::vector<std::string> fields = jsonCutParams.fields();
    MapCutParams mapCutParams;
    for (size_t i = 0; i < fields.size(); i++) {
        maps::json::Value jsonCutParamsItem = jsonCutParams[fields[i]];
        maps::json::Value jsonTile = jsonCutParamsItem[JSON_NAME_TILE];
        CutParams cutParams;
        cutParams.tileSize.width = jsonTile[JSON_NAME_WIDTH].as<int>();
        cutParams.tileSize.height = jsonTile[JSON_NAME_HEIGHT].as<int>();

        maps::json::Value jsonZooms = jsonCutParamsItem[JSON_NAME_ZOOMS];
        cutParams.photoSizes.resize(jsonZooms.size());
        for (size_t j = 0; j < jsonZooms.size(); j++) {
            maps::json::Value jsonZoom = jsonZooms[j];
            const int lvl = jsonZoom[JSON_NAME_LEVEL].as<int>();
            REQUIRE(0 <= lvl && lvl < (int)jsonZooms.size(),
                    "Invalid zoom level for cut params " << fields[i]);
            cutParams.photoSizes[lvl].width = jsonZoom[JSON_NAME_WIDTH].as<int>();
            cutParams.photoSizes[lvl].height = jsonZoom[JSON_NAME_HEIGHT].as<int>();
        }
        mapCutParams[stoll(fields[i])] = cutParams;
    }
    return mapCutParams;
}

std::vector<PanoramasFeature> loadPanoramasStreetFeatures(const maps::json::Value &jsonPanoramas) {
    static const std::string JSON_NAME_PRESET    = "preset";
    static const std::string JSON_NAME_MDSKEY    = "mds_key";
    static const std::string JSON_NAME_LON       = "lon";
    static const std::string JSON_NAME_LAT       = "lat";
    static const std::string JSON_NAME_CUTPARAMS = "cut_params";

    static const std::string VALID_PRESET = "street";

    std::vector<PanoramasFeature> result;
    for (size_t i = 0; i < jsonPanoramas.size(); i++) {
        maps::json::Value jsonPanoramasFeature = jsonPanoramas[i];
        if (!jsonPanoramasFeature.hasField(JSON_NAME_PRESET) ||
            VALID_PRESET != jsonPanoramasFeature[JSON_NAME_PRESET].as<std::string>() ||
            !jsonPanoramasFeature.hasField(JSON_NAME_MDSKEY) ||
            !jsonPanoramasFeature.hasField(JSON_NAME_LON) ||
            !jsonPanoramasFeature.hasField(JSON_NAME_LAT) ||
            !jsonPanoramasFeature.hasField(JSON_NAME_CUTPARAMS))
            continue;
        PanoramasFeature feature;
        feature.lon = jsonPanoramasFeature[JSON_NAME_LON].as<double>();
        feature.lat = jsonPanoramasFeature[JSON_NAME_LAT].as<double>();
        feature.mdsKey = jsonPanoramasFeature[JSON_NAME_MDSKEY].as<std::string>();
        feature.cutParamsID = jsonPanoramasFeature[JSON_NAME_CUTPARAMS].as<int64_t>();
        result.emplace_back(feature);
    }
    return result;
}

std::set<std::string> loadFilterMdsKeys(const std::string &filterPath) {
    std::set<std::string> filterMdsKeys;
    std::ifstream ifs(filterPath);
    if (!ifs.is_open())
        return filterMdsKeys;
    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        filterMdsKeys.insert(line);
    }
    return filterMdsKeys;
}

struct ImageData {
    PanoramasFeature feature;
    cv::Mat image;
};

struct ImageDataCounter {
    void IncCount(const ImageData&) {
        AtomicIncrement(Counter);
    }

    void DecCount(const ImageData&) {
        AtomicDecrement(Counter);
    }

    TAtomic Counter = 0;
};

typedef TLockFreeQueue<ImageData, ImageDataCounter> ImageQueue;

void fillFeaturesQueue(const maps::json::Value &jsonPanoramas, size_t fromIndex, size_t toIndex, const std::string &filterPath, ImageQueue &featureQueue) {
    std::vector<PanoramasFeature> panoramasFeatures = loadPanoramasStreetFeatures(jsonPanoramas);

    if (fromIndex >= panoramasFeatures.size()) {
        INFO() << "From index: " << fromIndex <<
            " great than panorams in description " << panoramasFeatures.size();
        return;
    }
    size_t toIndexP = std::min<size_t>(toIndex, panoramasFeatures.size());

    std::set<std::string> filterMdsKeys = loadFilterMdsKeys(filterPath);
    for (size_t i = fromIndex; i < toIndexP; i++) {
        const PanoramasFeature &feature = panoramasFeatures[i];
        if (filterMdsKeys.find(feature.mdsKey) != filterMdsKeys.end()) {
            INFO() << "mds_key: " << feature.mdsKey << " skipped (as already processed)";
            continue;
        }
        ImageData data;
        data.feature = feature;
        featureQueue.Enqueue(data);
    }
}

void parsePanoramasJson(const std::string &jsonPath, size_t fromIndex, size_t toIndex, const std::string &filterPath,
                        ImageQueue &featureQueue, MapCutParams &mapCutParams) {
    static const std::string JSON_NAME_CUTPARAMS = "cut_params";
    static const std::string JSON_NAME_PANORAMAS = "panoramas";

    maps::json::Value jsonMain = maps::json::Value::fromFile(jsonPath);
    mapCutParams = loadCutParams(jsonMain[JSON_NAME_CUTPARAMS]);
    fillFeaturesQueue(jsonMain[JSON_NAME_PANORAMAS], fromIndex, toIndex, filterPath, featureQueue);
}

class PanoramasImageDownloader
    : public IObjectInQueue {
public:
    PanoramasImageDownloader(ImageQueue* _featureQueue, ImageQueue* _imgQueue, const MapCutParams &_mapCutParams, size_t zoom)
        : featureQueue(_featureQueue)
        , imgQueue(_imgQueue)
        , mapCutParams(_mapCutParams)
        , zoomLevel(zoom)
        , running(true) {
    }
    void Process(void* /*threadSpecificResource*/) override {
        for (;;) {
            ImageData data;
            if (!featureQueue->Dequeue(&data))
                break;
            MapCutParams::const_iterator cit = mapCutParams.find(data.feature.cutParamsID);
            if (cit == mapCutParams.end()) {
                INFO() << "Unable to found cut params: " << data.feature.cutParamsID
                       << " for mds_key " << data.feature.mdsKey;
                continue;
            }
            data.image = downloadPhoto(data.feature.mdsKey, cit->second);
            imgQueue->Enqueue(data);
            wait();
        }
        running = false;
    }
    bool isRunning() const {
        return running;
    }
private:
    maps::http::Client client;
    ImageQueue* featureQueue;
    ImageQueue* imgQueue;
    const MapCutParams &mapCutParams;
    size_t zoomLevel;
    bool running;

    std::string getTileUrl(const std::string &mdsKey, int tileCol, int tileRow) {
        static const std::string URL_PREFIX = "http://s3.mds.yandex.net/pano/";
        std::stringstream ss;
        ss << URL_PREFIX << mdsKey << "/" << zoomLevel << "." << tileCol << "." << tileRow;
        return ss.str();
    }
    cv::Mat downloadTile(const std::string& url) {
        static const size_t RETRY_COUNT = 10;

        maps::common::RetryPolicy retryPolicy;
        retryPolicy.setTryNumber(RETRY_COUNT);

        auto validateResponse = [](const auto& maybeResponse) {
            return maybeResponse.valid() && maybeResponse.get().responseClass() != maps::http::ResponseClass::ServerError;
        };
        auto resp = maps::common::retry(
            [&]() {
            return maps::http::Request(client, maps::http::GET, maps::http::URL(url)).perform();
        },
            retryPolicy,
            validateResponse
            );
        if (resp.responseClass() != maps::http::ResponseClass::Success) {
            INFO() << "Unexpected response status " << resp.status() << " for url " << url;
            return cv::Mat();
        }

        const std::string body = resp.readBody();
        cv::Mat encImage((int)body.size(), 1, CV_8UC1, (void *)body.c_str());
        cv::Mat tile;
        try {
            tile = cv::imdecode(encImage, cv::IMREAD_COLOR | cv::IMREAD_IGNORE_ORIENTATION);
        }
        catch (...) {
            return cv::Mat();
        }
        return tile;
    }
    cv::Mat downloadPhoto(const std::string &mdsKey, const CutParams &cutParams) {
        const cv::Size &photoSize = cutParams.photoSizes[zoomLevel];
        const cv::Size &tileSize = cutParams.tileSize;
        const int cols = (photoSize.width + tileSize.width - 1) / tileSize.width;
        const int rows = (photoSize.height + tileSize.height - 1) / tileSize.height;

        cv::Mat image(photoSize, CV_8UC3);
        for (int row = 0; row < rows; row++) {
            cv::Range rowRange(row * tileSize.height, std::min((row + 1) * tileSize.height, photoSize.height));
            for (int col = 0; col < cols; col++) {
                cv::Range colRange(col * tileSize.width, std::min((col + 1) * tileSize.width, photoSize.width));
                cv::Mat tile = downloadTile(getTileUrl(mdsKey, col, row));
                if (tile.empty()) {
                    INFO() << "Unable to download image with mds_key: " << mdsKey;
                    return cv::Mat();
                }
                tile.copyTo(image(rowRange, colRange));
            }
        }
        return image;
    }

    void wait() {
        constexpr int PREFETCH_SIZE = 20;
        for (;;) {
            ImageDataCounter counter = imgQueue->GetCounter();
            if (PREFETCH_SIZE > (int)AtomicGet(counter.Counter))
                break;
            std::this_thread::sleep_for(THREAD_WAIT_TIMEOUT);
        }
    }
};

bool isDownloadersRunning(const std::vector<TAutoPtr<PanoramasImageDownloader>> &downloaders) {
    bool running = false;
    for (size_t i = 0; i < downloaders.size(); i++) {
        running |= downloaders[i]->isRunning();
    }
    return running;
}

void removeDuplicates(HouseNumberSigns &signs) {
    static const double COMPARE_THRESHOLD = 0.4;
    if (1 >= signs.size())
        return;

    std::sort(signs.begin(), signs.end(),
              [](const HouseNumberSign &a, const HouseNumberSign &b) { return a.box.area() > b.box.area();});

    for (size_t i = 0; i < signs.size() - 1; i++) {
        for (size_t j = i +  1; j < signs.size(); j++) {
            if ((signs[i].box & signs[j].box).area() > COMPARE_THRESHOLD * signs[j].box.area()) {
                signs.erase(signs.begin() + j);
                j--;
            }
        }
    }
}

HouseNumberSigns detectOnPhoto(const FasterRCNNDetector &detector, const cv::Mat &photo, size_t batchSize) {
    static const int TILE_SIZE = 1200;
    static const int TILE_MIN_OVERLAP = 30;

    REQUIRE(0 < batchSize, "Batch size should be great than zero");
    cv::Size photoSize = photo.size();
    std::vector<cv::Range> colsRange;
    std::vector<cv::Range> rowsRange;

    if (photoSize.width <= TILE_SIZE) {
        colsRange.push_back(cv::Range(0, photoSize.width));
    } else {
        int cellsCnt = (photoSize.width + TILE_SIZE - 2 * TILE_MIN_OVERLAP - 1) / (TILE_SIZE - TILE_MIN_OVERLAP);
        for (int i = 0; i < cellsCnt; i++) {
            const int s = (i == 0) ? 0 : (i * TILE_SIZE - i * (cellsCnt * TILE_SIZE - photoSize.width) / (cellsCnt - 1));
            colsRange.push_back(cv::Range(s, s + TILE_SIZE));
        }
    }
    if (photoSize.height <= TILE_SIZE) {
        rowsRange.push_back(cv::Range(0, photoSize.height));
    } else {
        int cellsCnt = (photoSize.height + TILE_SIZE - 2 * TILE_MIN_OVERLAP - 1) / (TILE_SIZE - TILE_MIN_OVERLAP);
        for (int i = 0; i < cellsCnt; i++) {
            const int s = (i == 0) ? 0 : (i * TILE_SIZE - i * (cellsCnt * TILE_SIZE - photoSize.height) / (cellsCnt - 1));
            rowsRange.push_back(cv::Range(s, s + TILE_SIZE));
        }
    }

    batchSize = std::min(batchSize, rowsRange.size() * colsRange.size());
    HouseNumberSigns signs;
    std::vector<cv::Mat> batch(batchSize);
    for (size_t i = 0; i < batchSize; i++) {
        batch[i] = cv::Mat(TILE_SIZE, TILE_SIZE, CV_8UC3);
    }
    std::vector<cv::Point> offsets(batchSize, cv::Point(0,0));
    size_t inBatch = 0;
    for (size_t row = 0; row < rowsRange.size(); row++) {
        for (size_t col = 0; col < colsRange.size(); col++) {
            photo(rowsRange[row], colsRange[col]).copyTo(batch[inBatch]);
            offsets[inBatch].x = colsRange[col].start;
            offsets[inBatch].y = rowsRange[row].start;
            inBatch++;
            if (inBatch == batch.size()) {
                std::vector<HouseNumberSigns> batchSigns = detector.detect(batch);
                for (size_t i = 0; i < batchSigns.size(); i++) {
                    const HouseNumberSigns &cellSigns = batchSigns[i];
                    const cv::Point &offset = offsets[i];
                    for (size_t j = 0; j < cellSigns.size(); j++) {
                        HouseNumberSign sign = cellSigns[j];
                        sign.box += offset;
                        signs.emplace_back(sign);
                    }
                }
                inBatch = 0;
            }
        }
    }
    if (0 < inBatch) {
        batch.resize(inBatch);
        std::vector<HouseNumberSigns> batchSigns = detector.detect(batch);
        for (size_t i = 0; i < batchSigns.size(); i++) {
            const HouseNumberSigns &cellSigns = batchSigns[i];
            const cv::Point &offset = offsets[i];
            for (size_t j = 0; j < cellSigns.size(); j++) {
                HouseNumberSign sign = cellSigns[j];
                sign.box += offset;
                signs.emplace_back(sign);
            }
        }
    }
    removeDuplicates(signs);
    return signs;
}

void expandRect(cv::Rect &rc, int minSize, int borderSize, const cv::Size &imageSize) {
    if (minSize >= imageSize.width) {
        rc.x = 0;
        rc.width = imageSize.width;
    }
    else {
        int border = (rc.width < minSize - 2 * borderSize) ? (minSize - rc.width) / 2 : borderSize;
        if (rc.x < border) {
            rc.width = std::min(std::max(minSize, rc.x + rc.width + borderSize), imageSize.width);
            rc.x = 0;
        }
        else if (rc.x + rc.width + border > imageSize.width) {
            rc.x = std::max(std::min(imageSize.width - minSize, rc.x - borderSize), 0);
            rc.width = imageSize.width - rc.x;
        }
        else {
            rc.x -= border;
            rc.width = std::min(std::max(minSize, rc.width + 2 * border), imageSize.width - rc.x);
        }
    }

    if (minSize >= imageSize.height) {
        rc.y = 0;
        rc.height = imageSize.height;
    }
    else {
        int border = (rc.height < minSize - 2 * borderSize) ? (minSize - rc.height) / 2 : borderSize;
        if (rc.y < border) {
            rc.height = std::min(std::max(minSize, rc.y + rc.height + borderSize), imageSize.height);
            rc.y = 0;
        }
        else if (rc.y + rc.height + border > imageSize.height) {
            rc.y = std::max(std::min(imageSize.height - minSize, rc.y - borderSize), 0);
            rc.height = imageSize.height - rc.y;
        }
        else {
            rc.y -= border;
            rc.height = std::min(std::max(minSize, rc.height + 2 * border), imageSize.height - rc.y);
        }
    }
}

void shiftSignsToRect(ClusterOfSigns &cluster) {
    const cv::Rect &rc = cluster.rc;
    HouseNumberSigns &signs = cluster.signs;
    for (size_t i = 0; i < signs.size(); i++) {
        signs[i].box.x -= rc.x;
        signs[i].box.y -= rc.y;
    }
}

std::vector<ClusterOfSigns> clusterizeSignsSet(const HouseNumberSigns &signs, const cv::Size &imageSize) {
    static const int CLUSTER_MIN_SIZE = 1200;
    static const int CLUSTER_MAX_SIZE = 1920;
    static const int CLUSTER_BORDER   = 30;

    REQUIRE(0 < signs.size(), "At least one sign must be in the set");

    ClusterOfSigns cluster;
    cluster.rc = signs[0].box;
    cluster.signs.push_back(signs[0]);
    for (size_t i = 1; i < signs.size(); i++) {
        cluster.rc = cluster.rc | signs[i].box;
        cluster.signs.push_back(signs[i]);
    }
    if ((cluster.rc.width < CLUSTER_MAX_SIZE - 2 * CLUSTER_BORDER) &&
        (cluster.rc.height < CLUSTER_MAX_SIZE - 2 * CLUSTER_BORDER)) {
        expandRect(cluster.rc, CLUSTER_MIN_SIZE, CLUSTER_BORDER, imageSize);
        std::vector<ClusterOfSigns> results(1, cluster);
        shiftSignsToRect(results[0]);
        return results;
    }
    // Most cases will be covered by above code, but if not we will make complex one
    std::vector<ClusterOfSigns> clusters;
    for (size_t i = 0; i < signs.size(); i++) {
        REQUIRE(signs[i].box.width < CLUSTER_MAX_SIZE - 2 * CLUSTER_BORDER,
                "Sign with width " << signs[i].box.width << " is too big");
        REQUIRE(signs[i].box.height < CLUSTER_MAX_SIZE - 2 * CLUSTER_BORDER,
                "Sign with height " << signs[i].box.height << " is too big");
        ClusterOfSigns cluster;
        cluster.rc = signs[i].box;
        cluster.signs.push_back(signs[i]);
        clusters.push_back(cluster);
    }
    std::vector<ClusterOfSigns> results;
    for (;0 < clusters.size();) {
        if (1 == clusters.size()) {
            results.push_back(clusters[0]);
            break;
        }
        size_t idx1 = 0;
        size_t idx2 = 1;
        cv::Rect rcmin = clusters[idx1].rc | clusters[idx2].rc;
        for (size_t i = 0; i < clusters.size() - 1; i++) {
            for (size_t j = i +  1; j < clusters.size(); j++) {
                cv::Rect rc = clusters[i].rc | clusters[j].rc;
                if (rc.area() < rcmin.area()) {
                    rcmin = rc;
                    idx1 = i;
                    idx2 = j;
                }
            }
        }

        if ((rcmin.width > CLUSTER_MAX_SIZE - 2 * CLUSTER_BORDER) ||
            (rcmin.height > CLUSTER_MAX_SIZE - 2 * CLUSTER_BORDER)) {
            results.insert(results.end(), clusters.begin(), clusters.end());
            break;
        }
        clusters[idx1].rc = clusters[idx1].rc | clusters[idx2].rc;
        clusters[idx1].signs.insert(clusters[idx1].signs.end(), clusters[idx2].signs.begin(), clusters[idx2].signs.end());
        clusters.erase(clusters.begin() + idx2);
    }

    for (size_t i = 0; i < results.size(); i++) {
        expandRect(results[i].rc, CLUSTER_MIN_SIZE, CLUSTER_BORDER, imageSize);
        shiftSignsToRect(results[i]);
    }
    return results;
}

void saveBBox(const cv::Rect& bbox, maps::json::ObjectBuilder &builder) {
    builder["bbox"] << [&bbox](maps::json::ArrayBuilder builder) {
        builder << [&bbox](maps::json::ArrayBuilder builder) {
                builder << bbox.x;
                builder << bbox.y;
            };
        builder << [&bbox](maps::json::ArrayBuilder builder) {
                builder << bbox.x + bbox.width;
                builder << bbox.y + bbox.height;
            };
    };
}

void saveObjects(const HouseNumberSigns& objects, maps::json::ObjectBuilder &builder) {
    static const std::string HOUSE_NUMBER_SIGN_TYPE_NAME = "house_number_sign";

    builder["objects"] << [&objects](maps::json::ArrayBuilder builder) {
        for (size_t i = 0; i < objects.size(); i++) {
                const HouseNumberSign &object = objects[i];
                builder << [&object](maps::json::ObjectBuilder builder) {
                builder["type"] = HOUSE_NUMBER_SIGN_TYPE_NAME;
                saveBBox(object.box, builder);
            };
        };
    };
}

void saveFeatureWithObjects(const PanoramasFeature &feature, const cv::Mat &image, const HouseNumberSigns &objects, maps::json::Builder &builder) {
    builder << [&feature, &objects, &image](maps::json::ObjectBuilder builder) {
        std::vector<uint8_t> encImage;
        cv::imencode(".jpg", image, encImage);

        builder["feature_id"] << (int64_t)-1;
        builder["source"]     << feature.mdsKey;
        builder["orientation"]<< (int)1;
        builder["image"] << maps::base64Encode(encImage);
        builder["lon"]<< feature.lon;
        builder["lat"]<< feature.lat;
        saveObjects(objects, builder);
    };
}

} //namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Launch house number signs detector on panoramas photos");

    maps::cmdline::Option<std::string> inputPath = parser.string("input")
        .required()
        .help("Path to description.json of panoramas");

    maps::cmdline::Option<std::string> outputPath = parser.string("output")
        .required()
        .help("Path to file for save results. Results for one image on the line in json format");

    maps::cmdline::Option<std::string> filterPath = parser.string("filter")
        .defaultValue("")
        .help("Path to text file with mds_key which already checked");

    maps::cmdline::Option<size_t> fromIndex = parser.size_t("from_idx")
        .defaultValue(0)
        .help("Index of start line in input file");

    maps::cmdline::Option<size_t> toIndex = parser.size_t("to_idx")
        .defaultValue(-1)
        .help("Index of line after last processed in input file");

    maps::cmdline::Option<size_t> zoom = parser.size_t("zoom")
        .defaultValue(1)
        .help("Zoom level of panoramas photo");

    maps::cmdline::Option<size_t> batchSize = parser.size_t("batch")
        .defaultValue(1)
        .help("Batch size (default: 1)");

    maps::cmdline::Option<size_t> threadsCount = parser.size_t("threads")
        .defaultValue(10)
        .help("Download threads count (default: 10)");

    parser.parse(argc, const_cast<char**>(argv));

    ImageQueue featureQueue, imageQueue;
    MapCutParams mapCutParams;
    parsePanoramasJson(inputPath, fromIndex, toIndex, filterPath, featureQueue, mapCutParams);

    TAutoPtr<IThreadPool> mtpQueue = CreateThreadPool(threadsCount);
    std::vector<TAutoPtr<PanoramasImageDownloader>> downloaders;
    for (size_t i = 0; i < threadsCount; i++) {
        TAutoPtr<PanoramasImageDownloader> downloader(new PanoramasImageDownloader(&featureQueue, &imageQueue, mapCutParams, zoom));
        mtpQueue->SafeAdd(downloader.Get());
        downloaders.push_back(downloader);
    }

    std::ofstream reworked(filterPath, std::ofstream::out | std::ofstream::app);
    std::ofstream ofs(outputPath, std::ofstream::out | std::ofstream::app);
    FasterRCNNDetector detector;
    for (;;) {
        ImageData data;
        if (imageQueue.Dequeue(&data)) {
            if (data.image.empty())
                continue;
            INFO() << "mds_key: " << data.feature.mdsKey << " detect on photo";
            HouseNumberSigns signs = detectOnPhoto(detector, data.image, batchSize);

            if (0 < signs.size()) {
                std::vector<ClusterOfSigns> clusters = clusterizeSignsSet(signs, data.image.size());
                for (size_t i = 0; i < clusters.size(); i++) {
                    std::stringstream jsonLine;
                    maps::json::Builder builder(jsonLine);
                    saveFeatureWithObjects(data.feature, data.image(clusters[i].rc), clusters[i].signs, builder);
                    ofs << jsonLine.str() << std::endl;
                }
                INFO() << "mds_key: " << data.feature.mdsKey
                    << " signs count " << signs.size()
                    << " in " << clusters.size() << " clusters";
            }
            else {
                INFO() << "mds_key: " << data.feature.mdsKey << " signs count " << signs.size();
            }
            reworked << data.feature.mdsKey << std::endl;
        } else if (isDownloadersRunning(downloaders)) {
            std::this_thread::sleep_for(THREAD_WAIT_TIMEOUT);
        }
        else {
            break;
        }
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
