#include "yt.h"
#include "detector.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/object_in_photo_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/io.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/operation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/retry.h>

#include <maps/libs/log8/include/log8.h>
#include <mapreduce/yt/interface/client.h>
#include <util/generic/size_literals.h>

#include <algorithm>

namespace maps::mrc::image_analyzer {

namespace {

constexpr std::size_t MAX_LOAD_JOBS_COUNT = 300;
constexpr std::size_t MAX_DETECTION_JOBS_COUNT = 5000;
constexpr std::size_t MIN_IMAGES_PER_JOB = 30;

template <typename Functor>
auto retry(Functor&& func) -> decltype(func())
{
    return common::retryOnException<std::exception>(
        common::RetryPolicy()
            .setInitialTimeout(std::chrono::seconds(1))
            .setMaxAttempts(4)
            .setTimeoutBackoff(1),
        std::forward<Functor>(func));
}

class YtProcessor
{
public:
    explicit YtProcessor(const common::Config& mrcConfig)
        : mrcConfig_(mrcConfig)
        , client_(mrcConfig.externals().yt().makeClient())
    {
        const TString basePath(mrcConfig_.externals().yt().path() + "/privacy-detector-tool");

        if (!client_->Exists(basePath)) {
            INFO() << "Create YT node " << basePath;
            client_->Create(basePath, NYT::NT_MAP, NYT::TCreateOptions()
                .Recursive(true));
        }
        inputPath_ = basePath + "/features-input";
        outputPath_ = basePath + "/image-objects-output";
    }

    db::ObjectsInPhoto process(const db::Features& features)
    {
        INFO() << "YT: Store " << features.size() << " features at " << inputPath_;
        yt::saveToTable(*client_, inputPath_, features);

        INFO() << "YT: Load feature images";
        yt::uploadFeatureImage(
            mrcConfig_, *client_, inputPath_, inputPath_, MAX_LOAD_JOBS_COUNT);

        size_t numRows = yt::getRowCount(*client_, inputPath_);
        size_t jobsCount = std::clamp(numRows / MIN_IMAGES_PER_JOB,
                                      1UL, MAX_DETECTION_JOBS_COUNT);

        INFO() << "YT: Detect privacy objects";
        runMapper(jobsCount);

        INFO() << "YT: Load results";
        auto results = yt::loadFromTable<db::ObjectsInPhoto>(*client_, outputPath_);

        client_->Remove(inputPath_);
        client_->Remove(outputPath_);

        INFO() << "YT: Done";
        return results;
    }

private:
    void runMapper(size_t jobsCount)
    {
        NYT::TUserJobSpec jobSpec;
        jobSpec.MemoryLimit(3_GB);

        const auto operationOptions = NYT::TNode::CreateMap()
            ("title", "privacy objects detection tool")
            ("mapper", NYT::TNode::CreateMap()
                           ("memory_limit", 6_GB)
                           ("memory_reserve_factor", 0.6));

        client_->Map(
            NYT::TMapOperationSpec()
                .AddInput<NYT::TNode>(inputPath_)
                .AddOutput<NYT::TNode>(outputPath_)
                .JobCount(jobsCount)
                .MapperSpec(jobSpec),
            new PrivacyObjectsDetector(),
            NYT::TOperationOptions().Spec(operationOptions)
        );
    }

private:
    const common::Config& mrcConfig_;
    NYT::IClientPtr client_;
    TString inputPath_;
    TString outputPath_;
};

} // namespace


db::ObjectsInPhoto detectObjectsOnYT(
    const db::Features& features,
    const common::Config& mrcConfig)
{
    YtProcessor ytProcessor(mrcConfig);
    return retry([&]{ return ytProcessor.process(features); });
}

} // maps::mrc::image_analyzer
