import argparse
import math

import yt.wrapper as yt
import yt.yson as yson
import yandex.maps.geolib3 as geolib


def view(view_func):
    def decorated(obj, **kwargs):
        return yson.dumps(yson.to_yson_type(view_func(obj, **kwargs)))

    return decorated


@view
def feature_view(feature):
    return {
        'lon': feature['lon'],
        'lat': feature['lat'],
        'text': feature['text'],
        '_pointstyle': feature['pointstyle'],
    }


@view
def feature_head_view(feature):
    start = geolib.Point2(feature['lon'], feature['lat'])

    end = geolib.geo_to_mercator(start)

    angle = math.pi / 2 - feature['heading'] / 180 * math.pi
    end = geolib.Point2(end.x + 7 * math.cos(angle), end.y + 7 * math.sin(angle))

    end = geolib.mercator_to_geo(end)

    return {
        'start_lon': start.x,
        'start_lat': start.y,
        'end_lon': end.x,
        'end_lat': end.y,
        '_linestyle': feature['linestyle'],
    }


def get_feature(pos, heading, feature_id, color):
    if isinstance(pos, list):
        lat, lon, _ = pos
    else:
        point = geolib.Point2(pos['x'], pos['y'])
        point = geolib.mercator_to_geo(point)
        lon, lat = point.x, point.y
    return {
        'lon': lon,
        'lat': lat,
        'heading': heading,
        'text': '<a target=\'_blank\' href=\'https://n.maps.yandex.ru/#!/mrc/{id}/\'>{id}</a>'.format(id=feature_id),
        'pointstyle': '{color}:{color}:2'.format(color=color),
        'linestyle': '{color}:1'.format(color=color),
    }


def prepare_features(row, show_ground_truth=False):
    features = []
    tracked_by_sfm = False

    if 'true_pos' in row and show_ground_truth:
        features.append(get_feature(row['true_pos'], row['true_heading'], row['feature_id'], color='cyan'))

    if row.get('refined_pos', None) is not None:
        tracked_by_sfm = True
        features.append(get_feature(row['refined_pos'], row['refined_heading'], row['feature_id'], color='green'))

    features.append(get_feature(row['pos'], row['heading'], row['feature_id'], color='black' if tracked_by_sfm else 'red'))

    return features


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--path',
        help='Path to yt table with lon, lat and ground truth lon lat',
        required=True,
        type=str,
    )
    parser.add_argument(
        '--frames',
        help='Path to file with frame ids to visualize',
        type=str,
    )
    parser.add_argument(
        '--show-ground-truth',
        help='show ground truth points (if available)',
        action='store_true',
    )

    args = parser.parse_args()
    client = yt.YtClient('hahn')
    ids = set()

    if args.frames is not None:
        with open(args.frames) as f:
            ids = set(map(int, f.readlines()))

    for row in yt.read_table(args.path, client=client):
        if len(ids) and row['feature_id'] not in ids:
            continue

        features = prepare_features(row, args.show_ground_truth)
        for feature in features:
            print(feature_view(feature))
            print(feature_head_view(feature))

if __name__ == "__main__":
    main()
