#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <opencv2/opencv.hpp>

#include <string>
#include <vector>

namespace {
    static void normalizeImagesPair(cv::Mat &img0, cv::Mat &img1) {
        CV_Assert(img0.rows == img1.rows && img0.cols == img1.cols);
        CV_Assert(img0.type() == CV_8UC1);
        CV_Assert(img1.type() == CV_8UC1);

        double minVal0, maxVal0, minVal1, maxVal1;
        cv::minMaxLoc(img0, &minVal0, &maxVal0);
        cv::minMaxLoc(img1, &minVal1, &maxVal1);

        const double minVal = MIN(minVal0, minVal1);
        const double maxVal = MAX(maxVal0, maxVal1);

        const double delta = maxVal - minVal;
        if (delta < DBL_EPSILON) return;

        const int rows = img0.rows;
        const int cols = img0.cols;
        for (int row = 0; row < rows; row++) {
            uchar *ptrImg0 = img0.ptr(row);
            uchar *ptrImg1 = img1.ptr(row);
            for (int col = 0; col < cols; col++) {
                ptrImg0[col] = (uchar)(255. * (ptrImg0[col] - minVal) / delta);
                ptrImg1[col] = (uchar)(255. * (ptrImg1[col] - minVal) / delta);
            }
        }
    }
} //namespace


int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;
    auto inputFilterParam = parser.string("input_filter")\
        .required()\
        .help("filter path for input images");

    auto outputFolderPathParam = parser.string("output_folder")\
        .required()\
        .help("path to output folder");

    parser.parse(argc, argv);

    std::vector<cv::String> files;
    cv::glob(inputFilterParam, files);
    if (files.size() <= 1)
        ERROR() << "Too few files";

    cv::Size frame_size;
    cv::Ptr<cv::DenseOpticalFlow> dof = cv::optflow::createOptFlow_DeepFlow();
    for (size_t i = 0; i < files.size() - 1; i++) {
        INFO() << "Rework file: " << files[i];

        cv::Mat img0 = cv::imread(files[i], cv::IMREAD_GRAYSCALE);
        cv::Mat img1 = cv::imread(files[i + 1], cv::IMREAD_GRAYSCALE);

        cv::resize(img0, img0, cv::Size(), 0.5, 0.5);
        cv::resize(img1, img1, cv::Size(), 0.5, 0.5);

        normalizeImagesPair(img0, img1);

        if (frame_size.area() == 0) {
            frame_size = img0.size();
            frame_size.height *= 2;
        }

        cv::Mat flow;
        dof->calc(img0, img1, flow);

        cv::Mat flow_split[2];
        cv::Mat magnitude, angle;
        cv::split(flow, flow_split);
        cv::cartToPolar(flow_split[0], flow_split[1], magnitude, angle);

        cv::Mat flow_show;
        magnitude.convertTo(flow_show, CV_8U, 2.);

        const double mean = cv::mean(magnitude)[0];
        cv::Mat frame(frame_size, CV_8UC3);
        cv::cvtColor(img1, frame.rowRange(0, img1.rows), cv::COLOR_GRAY2BGR);
        cv::cvtColor(flow_show, frame.rowRange(img1.rows, frame_size.height), cv::COLOR_GRAY2BGR);

        cv::String text = cv::format("mean OF: %0.2f", mean);
        cv::Scalar color = (mean < 5.) ? cv::Scalar(0, 255, 0) : cv::Scalar(0, 0, 255);
        cv::putText(frame, text, cv::Point(10, 20), 0, 0.5, color, 2);

        size_t pos = files[i].find_last_of("/\\");
        std::string filename = files[i];
        if (std::string::npos != pos)
            filename = files[i].substr(pos);
        filename = (std::string)outputFolderPathParam + "/" + filename;
        cv::imwrite(filename, frame);
    }

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
