#include <maps/wikimap/mapspro/services/mrc/libs/classifiers/include/rotation_classifier.h>
#include <maps/wikimap/mapspro/services/mrc/libs/signdetect/include/signdetect_faster_rcnn.h>
#include <maps/wikimap/mapspro/services/mrc/libs/signdetect/include/detected_sign.h>
#include <maps/wikimap/mapspro/services/mrc/libs/sideview_classifier/include/sideview.h>
#include <maps/wikimap/mapspro/services/mrc/libs/carsegm/include/carsegm.h>
#include <maps/wikimap/mapspro/services/mrc/libs/privacy_detector/include/privacy_detector_faster_rcnn.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <opencv2/opencv.hpp>

#include <fstream>
#include <iostream>
#include <sstream>
#include <vector>
#include <chrono>

using namespace std::chrono;
using namespace maps::mrc::sideview;
using namespace maps::mrc::classifiers;
using namespace maps::mrc::common;
namespace tsd = maps::mrc::signdetect;
using namespace maps::mrc::carsegm;
namespace pd = maps::mrc::privacy_detector;

namespace {

std::vector<std::string> loadImagePathList(const std::string &path) {
    std::vector<std::string> result;

    std::ifstream ifs(path);
    if (!ifs.is_open())
        return result;

    for (; !ifs.eof();) {
        std::string line; std::getline(ifs, line);
        if (line.empty())
            continue;
        if ('#' == line[0])
            continue;
        result.emplace_back(line);
    }
    return result;
}

void applyOrientation(cv::Mat &image, const ImageOrientation &orientation){
    switch ((int)orientation) {
    case 2:
        cv::flip(image, image, 1);
        break;
    case 3:
        cv::flip(image, image, -1);
        break;
    case 4:
        cv::flip(image, image, 0);
        break;
    case 5:
        image = image.t();
        break;
    case 6:
        image = image.t();
        cv::flip(image, image, 1);
        break;
    case 7:
        image = image.t();
        cv::flip(image, image, -1);
        break;
    case 8:
        image = image.t();
        cv::flip(image, image, 0);
        break;
    }
}
} //namespace

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Test time of classifiers. Classifier of image rotation enable always");

    maps::cmdline::Option<std::string> inputPath = parser.string("input")
        .required()
        .help("Path to list of image files");

    maps::cmdline::Option<bool> enableSVParam = parser.flag("sideview")
        .help("check sideview classifier")
        .defaultValue(false);

    maps::cmdline::Option<bool> enableTSDetectorParam = parser.flag("tsdetect")
        .help("check traffic signs detector")
        .defaultValue(false);

    maps::cmdline::Option<bool> enableCarSegmentatorParam = parser.flag("carsegm")
        .help("check traffic signs detector")
        .defaultValue(false);

    maps::cmdline::Option<bool> enablePrivacyDetectorParam = parser.flag("privdetect")
        .help("check privacy detector")
        .defaultValue(false);

    maps::cmdline::Option<bool> enableAll = parser.flag("all")
        .help("check all detectors and classifiers")
        .defaultValue(false);

    parser.parse(argc, const_cast<char**>(argv));

    bool enableSV = enableSVParam;
    bool enableTSDetector = enableTSDetectorParam;
    bool enableCarSegmentator = enableCarSegmentatorParam;
    bool enablePrivacyDetector = enablePrivacyDetectorParam;

    if (enableAll) {
        enableSV = true;
        enableTSDetector = true;
        enableCarSegmentator = true;
        enablePrivacyDetector = true;
    }

    std::vector<std::string> imageList = loadImagePathList(inputPath);
    REQUIRE(imageList.size() > 1, "Must be more than 1 image in list");

    RotationClassifier rotClassifier;
    tsd::FasterRCNNDetector tsDetector;
    CarSegmentator carSegmentator;
    pd::FasterRCNNDetector pdDetector;

    tsd::DetectedSigns signs;
    ImageOrientation orientation;
    cv::Mat cars;
    pd::PrivacyImageBoxes pdBoxes;

    high_resolution_clock::time_point start, finish;
    if (enableSV) {
        SideViewClassifier svClassifier;
        std::pair<SideViewType, float> svResults;

        start = high_resolution_clock::now();

        cv::Mat currentImage = cv::imread(imageList[0], cv::IMREAD_COLOR + cv::IMREAD_IGNORE_ORIENTATION);
        orientation = rotClassifier.detectImageOrientation(currentImage);
        applyOrientation(currentImage, orientation);
        if (enableTSDetector)
            signs = tsDetector.detect(currentImage);
        if (enableCarSegmentator)
            cars = carSegmentator.segment(currentImage);
        if (enablePrivacyDetector)
            pdBoxes = pdDetector.detect(currentImage);

        for (size_t i = 1; i < imageList.size(); i++) {
            cv::Mat nextImage = cv::imread(imageList[i], cv::IMREAD_COLOR + cv::IMREAD_IGNORE_ORIENTATION);
            orientation =  rotClassifier.detectImageOrientation(nextImage);
            applyOrientation(nextImage, orientation);
            if (enableTSDetector)
                signs = tsDetector.detect(nextImage);
            if (enableCarSegmentator)
                cars = carSegmentator.segment(currentImage);
            if (enablePrivacyDetector)
                pdBoxes = pdDetector.detect(currentImage);
            svResults = svClassifier.inference(currentImage, nextImage);
            currentImage = nextImage;
        }
        finish = high_resolution_clock::now();
    } else {
        start = high_resolution_clock::now();

        for (size_t i = 0; i < imageList.size(); i++) {
            cv::Mat image = cv::imread(imageList[i], cv::IMREAD_COLOR + cv::IMREAD_IGNORE_ORIENTATION);

            ImageOrientation orientation =  rotClassifier.detectImageOrientation(image);
            applyOrientation(image, orientation);
            if (enableTSDetector)
                signs = tsDetector.detect(image);
            if (enableCarSegmentator)
                cars = carSegmentator.segment(image);
            if (enablePrivacyDetector)
                pdBoxes = pdDetector.detect(image);
        }
        finish = high_resolution_clock::now();
    }
    duration<double> elapsed = finish - start;
    INFO() << "Elapsed time: " << elapsed.count() << " sec";

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
