#include "tools.h"

#include <maps/libs/common/include/make_batches.h>
#include <maps/libs/http/include/http.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/retry.h>

#include <tbb/tbb.h>

#include <boost/range/algorithm_ext/erase.hpp>

#include <QImage>

#include <exception>

namespace maps {
namespace mrc {
namespace photo {
namespace {

template <typename Functor>
auto retry(Functor&& func) -> decltype(func())
{
    return common::retryOnException<std::exception>(
        common::RetryPolicy()
            .setInitialTimeout(std::chrono::seconds(1))
            .setMaxAttempts(6)
            .setTimeoutBackoff(2),
        func);
}

class DbWrapper {
public:
    explicit DbWrapper(pgpool3::Pool& pool) : pool_(pool) {}

    db::TIds loadFeaturesIds()
    {
        return retry([&] {
            return db::Gateway{*pool_.slaveTransaction()}.loadFeaturesIds();
        });
    }

    db::Features loadFeaturesByIds(const db::TIds& ids)
    {
        return retry([&] {
            return db::Gateway{*pool_.slaveTransaction()}.loadFeaturesByIds(
                ids);
        });
    }

    void save(db::Features& features)
    {
        retry([&] {
            auto txn = pool_.masterWriteableTransaction();
            db::Gateway{*txn}.save(features);
            txn->commit();
        });
    }

private:
    pgpool3::Pool& pool_;
};

class MdsWrapper {
public:
    explicit MdsWrapper(mds::Mds& mds) : mds_(mds) {}

    void fillSizes(db::Features& features)
    {
        tbb::parallel_for((size_t)0, features.size(), [&](size_t i) {
            try {
                retry([&] { fillSize(features[i]); });
            }
            catch (const std::exception& e) {
                WARN() << e.what() << "; id: " << features[i].id();
            }
        });
    }

private:
    mds::Mds& mds_;
    maps::http::Client httpClient_;

    void fillSize(db::Feature& feature)
    {
        auto url = mds_.makeReadUrl(
            maps::mds::Key{feature.mdsGroupId(), feature.mdsPath()});
        maps::http::URL urlWrapper{url};
        maps::http::Request proxyRequest(httpClient_, maps::http::GET, urlWrapper);
        auto proxyResponse = proxyRequest.perform();
        REQUIRE(proxyResponse.status() == 200, "Invalid MDS response");
        std::string buffer = proxyResponse.readBody();
        QImage image;
        if (image.loadFromData((const uchar*)buffer.data(),
                               static_cast<int>(buffer.size()))) {
            feature.setWidth(image.width());
            feature.setHeight(image.height());
        }
        else {
            WARN() << "Invalid image; id: " << feature.id();
        }
    }
};

} // anonymous namespace

void fillSizes(pgpool3::Pool& pool, mds::Mds& mds, size_t batchSize)
{
    DbWrapper dbWrapper{pool};
    MdsWrapper mdsWrapper{mds};
    auto ids = dbWrapper.loadFeaturesIds();
    INFO() << "loaded IDs: " << ids.size();
    size_t loaded = 0;
    size_t saved = 0;
    for (const auto& batch : maps::common::makeBatches(ids, batchSize)) {
        auto features
            = dbWrapper.loadFeaturesByIds({batch.begin(), batch.end()});
        loaded += features.size();
        boost::remove_erase_if(features, [](const db::Feature& feature) {
            return feature.width() && feature.height();
        });
        if (features.empty())
            continue;
        mdsWrapper.fillSizes(features);
        boost::remove_erase_if(features, [](const db::Feature& feature) {
            return !feature.width() || !feature.height();
        });
        if (features.empty())
            continue;
        dbWrapper.save(features);
        saved += features.size();
        INFO() << "Features: loaded " << loaded << ", saved " << saved;
    }
    INFO() << "Done. Features: loaded " << loaded << ", saved " << saved;
}

} // photo
} // mrc
} // maps
