#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point_gateway.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

namespace mrc = maps::mrc;
namespace db = maps::mrc::db;

namespace {

db::Features loadFeatures(
        maps::pgpool3::Pool& pool,
        const std::string& sourceId,
        maps::chrono::TimePoint minTimestamp,
        maps::chrono::TimePoint maxTimestamp,
        db::TId minFeatureId,
        db::TId maxFeatureId)
{
    auto txn = pool.slaveTransaction();

    INFO() << "Load all features by filter";
    db::FeatureGateway gateway(*txn);

    auto features = gateway.load(
         db::table::Feature::sourceId.equals(sourceId) &&
         db::table::Feature::date.between(minTimestamp, maxTimestamp) &&
         db::table::Feature::id.between(minFeatureId, maxFeatureId));

    return features;
}

db::TrackPoints loadTrackPoints(
        maps::pgpool3::Pool& pool,
        const std::string& sourceId,
        maps::chrono::TimePoint minTimestamp,
        maps::chrono::TimePoint maxTimestamp,
        db::TId minId,
        db::TId maxId)
{
    auto txn = pool.slaveTransaction();

    INFO() << "Load all track points by filter";
    db::TrackPointGateway gateway(*txn);

    auto trackPoints = gateway.load(
         db::table::TrackPoint::sourceId.equals(sourceId) &&
         db::table::TrackPoint::timestamp.between(minTimestamp, maxTimestamp) &&
         db::table::TrackPoint::id.between(minId, maxId));

    return trackPoints;
}

void shiftFeaturesDate(
        pqxx::transaction_base& txn,
        db::Features& features,
        int shiftSec)
{
    for (auto& feature: features) {
        auto date = feature.timestamp();
        feature.setTimestamp(date + std::chrono::seconds(shiftSec));
        DEBUG() << "Feature " << feature.id()
               << ": old date " << maps::chrono::formatIsoDateTime(date)
               << ", new date " << maps::chrono::formatIsoDateTime(feature.timestamp());
    }

    db::FeatureGateway gateway(txn);
    gateway.update(features, db::UpdateFeatureTxn::Yes);
}


void shiftTrackPointsDate(
        pqxx::transaction_base& txn,
        db::TrackPoints& trackPoints,
        int shiftSec)
{
    for (auto& tp: trackPoints) {
        auto date = tp.timestamp();
        tp.setTimestamp(date + std::chrono::seconds(shiftSec));
        DEBUG() << "Track point " << tp.id()
               << ": old date " << maps::chrono::formatIsoDateTime(date)
               << ", new date " << maps::chrono::formatIsoDateTime(tp.timestamp());
    }

    db::TrackPointGateway gateway(txn);
    gateway.update(trackPoints);
}

} // namespace

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser(
        "Tool for shifting date of photos and track points");
    auto configPath = parser.string("config")
        .help("path to configuration");
    auto secretVersion = parser.string("secret-version")
        .help("version for secrets from yav.yandex-team.ru");

    auto sourceId = parser.string("source-id")
        .help("source_id of photos").required();

    auto minDate = parser.string("min-date")
        .help("minimal date").required();
    auto maxDate = parser.string("max-date")
        .help("maximal date").required();

    auto minFeatureId = parser.num("min-feature-id")
        .help("min feature id").required();
    auto maxFeatureId = parser.num("max-feature-id")
        .help("max feature id").required();

    auto minTrackId = parser.num("min-track-id")
        .help("min track point id").required();
    auto maxTrackId = parser.num("max-track-id")
        .help("max track point id").required();

    auto shiftSec = parser.num("shift-sec").required();

    auto isDryRun = parser.flag("dry-run")
        .help("don't modify db")
        .defaultValue(false);

    parser.parse(argc, argv);

    auto config =  maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);
    auto poolHolder = config.makePoolHolder();

    auto features = loadFeatures(
        poolHolder.pool(),
        sourceId,
        maps::chrono::parseIsoDateTime(minDate),
        maps::chrono::parseIsoDateTime(maxDate),
        minFeatureId,
        maxFeatureId
    );
    INFO() << "Loaded " << features.size() << " features";

    auto trackPoints = loadTrackPoints(
        poolHolder.pool(),
        sourceId,
        maps::chrono::parseIsoDateTime(minDate),
        maps::chrono::parseIsoDateTime(maxDate),
        minTrackId,
        maxTrackId
    );
    INFO() << "Loaded " << trackPoints.size() << " trask points";

    auto txn = poolHolder.pool().masterWriteableTransaction();

    shiftFeaturesDate(*txn, features, shiftSec);
    shiftTrackPointsDate(*txn, trackPoints, shiftSec);

    if (!isDryRun) {
        txn->commit();
        INFO() << "Committed!";
    } else {
        INFO() << "Skip commit in dry-run mode";
    }

    INFO() << "Done";
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
