#include "tool.h"

#include <maps/libs/sql_chemistry/include/batch_load.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/parallel_for_each.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/libs/log8/include/log8.h>

namespace maps::mrc {
namespace {

void parallelFillPrivacy(db::Features& features,
                         privacy::RegionPrivacy& regionPrivacy)
{
    common::parallelForEach(features.begin(), features.end(),
        [&](auto&, auto& feature) {
            if (!feature.hasPos()) {
                WARN() << "Feature " << feature.id() << " has no position";
                return;
            }

            auto privacy = regionPrivacy.evalFeaturePrivacy(feature.geodeticPos());
            INFO() << "Set privacy `" << privacy << "` to feature " << feature.id();
            feature.setPrivacy(privacy);
        });
}

void save(pgpool3::Pool& pool, TArrayRef<db::Feature> features)
{
    size_t SAVE_BATCH_SIZE{1000}, pos{0};

    while (pos < features.size()) {
        auto len = std::min(SAVE_BATCH_SIZE, features.size() - pos);
        TArrayRef<db::Feature> batch(features.data() + pos, len);
        auto txn = pool.masterWriteableTransaction();
        db::FeatureGateway{*txn}.update(batch, db::UpdateFeatureTxn::Yes);
        txn->commit();
        pos += len;
    }
}

sql_chemistry::FiltersCollection
featureIdRangeFilter(db::TId minFeatureId, db::TId maxFeatureId)
{
    sql_chemistry::FiltersCollection filter(sql_chemistry::op::Logical::And);
    if (minFeatureId) {
        filter.add(db::table::Feature::id >= minFeatureId);
    }
    if (maxFeatureId) {
        filter.add(db::table::Feature::id <= maxFeatureId);
    }
    return filter;
}

sql_chemistry::FiltersCollection
featureIdsFilter(const db::TIds& featureIds)
{
    sql_chemistry::FiltersCollection filter(sql_chemistry::op::Logical::And);
    filter.add(db::table::Feature::id.in(featureIds));
    return filter;
}

void fillPrivacyImpl(pgpool3::Pool& pool,
                     privacy::RegionPrivacy& regionPrivacy,
                     const sql_chemistry::Filter& filter,
                     size_t batchSize,
                     bool dryRun)
{
    db::Features features;
    features.reserve(batchSize);
    size_t processed{0};

    auto txn = pool.slaveTransaction();

    sql_chemistry::BatchLoad<db::table::Feature> batches{batchSize, filter};

    INFO() << "Start processing features...";
    while (batches.next(*txn)) {
        for (auto& feature : batches) {
            features.push_back(std::move(feature));
        }

        parallelFillPrivacy(features, regionPrivacy);

        if (!dryRun) {
            save(pool, features);
        } else {
            INFO() << "Skip saving features in dry-run mode";
        }

        processed += features.size();
        INFO() << "Processed: " << processed << ", up to ID " << features.back().id();
        features.clear();
    }
}

} // anonymous namespace


void fillPrivacy(pgpool3::Pool& pool,
                 privacy::RegionPrivacy& regionPrivacy,
                 size_t minFeatureId,
                 size_t maxFeatureId,
                 size_t batchSize,
                 bool dryRun)
{
    auto filter = featureIdRangeFilter(minFeatureId, maxFeatureId);
    fillPrivacyImpl(pool, regionPrivacy, filter, batchSize, dryRun);
}

void fillPrivacy(pgpool3::Pool& pool,
                 privacy::RegionPrivacy& regionPrivacy,
                 const db::TIds& featureIds,
                 size_t batchSize,
                 bool dryRun)
{
    auto filter = featureIdsFilter(featureIds);
    fillPrivacyImpl(pool, regionPrivacy, filter, batchSize, dryRun);
}

} // maps::mrc
