#include "tool.h"

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/sql_chemistry/include/system_information.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/make_batches.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

namespace db = maps::mrc::db;

namespace {

maps::mrc::tool::Mode parseMode(const std::string& mode)
{
    if (mode == "photo") {
        return maps::mrc::tool::Mode::Photo;
    }
    else if (mode == "ride") {
        return maps::mrc::tool::Mode::Ride;
    }
    else if (mode == "user") {
        return maps::mrc::tool::Mode::User;
    }
    else {
        throw maps::Exception{} << "Invalid mode: " << mode;
    }
}

} // anonymous namespace

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser("tool for unpublishing photos");
    auto configPath = parser.string("config").help("path to configuration");
    auto mode = parser.string("mode").defaultValue("ride").help(
        "unpublishing mode: photo | ride (default) | user");
    auto photoId = parser.num("photo").help("photo ID to unpublish");
    auto maxPhotoId
        = parser.num("max-photo")
              .defaultValue(0)
              .help("upper bound photo ID to unpublish (for ride mode only)");
    auto dryRun = parser.flag("dry-run").defaultValue(false).help(
        "test run in read-only mode");
    parser.parse(argc, argv);

    auto cfg = maps::mrc::common::templateConfigFromCmdPath(configPath);
    auto pool = cfg.makePoolHolder();
    auto photos = maps::mrc::tool::load(*pool.pool().slaveTransaction(),
                                        parseMode(mode), photoId, maxPhotoId);
    INFO() << "loaded photos to unpublish: " << photos.size();

    if (!dryRun) {
        size_t counter = 0;
        for (auto& batch : maps::common::makeBatches(photos, 100)) {
            db::Features features{batch.begin(), batch.end()};
            for (auto& feature : features) {
                feature.setAutomaticShouldBePublished(false);
            }
            auto txn = pool.pool().masterWriteableTransaction();

            db::FeatureGateway{*txn}.update(features, db::UpdateFeatureTxn::Yes);
            txn->commit();
            counter += features.size();
            INFO() << "upublished " << counter << " photos";
        }
        INFO() << "done";
    }
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
