#include <maps/wikimap/mapspro/services/mrc/tools/photos_unpublisher/tool.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/libs/geolib/include/vector.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

namespace maps {
namespace mrc {
namespace tool {
namespace tests {
namespace {

db::Feature makeFeature(const std::string& sourceId,
                        const geolib3::Point2& basePos,
                        chrono::TimePoint baseTimePoint,
                        size_t featureNum)
{
    return sql_chemistry::GatewayAccess<db::Feature>::construct()
        .setSourceId(sourceId)
        .setGeodeticPos(basePos
                        + geolib3::Vector2{0.0000134, -0.000016}
                              * double(featureNum))
        .setHeading(geolib3::Heading{170.0})
        .setTimestamp(baseTimePoint + std::chrono::seconds(featureNum - 1))
        .setMdsKey({"1234", "abc"})
        .setDataset(db::Dataset::Rides)
        .setSize(6, 9)
        .setAutomaticShouldBePublished(true)
        .setIsPublished(true);
}

} // anonymous namespace

struct UnpublishTest : testing::Test {
    using TestFixture
        = unittest::WithUnittestConfig<unittest::DatabaseFixture>;

    UnpublishTest() { testFixture_.reset(new TestFixture); }

    db::Features fillData()
    {
        const geolib3::Point2 BASE_FEATURE_POS{37.547148, 55.727823};
        const chrono::TimePoint BASE_TIME_POINT
            = chrono::parseSqlDateTime("2017-05-17 11:03:17+03");
        db::Features result;

        // first passage
        for (size_t featureNum : {1, 2, 3}) {
            result.push_back(makeFeature("007", BASE_FEATURE_POS,
                                         BASE_TIME_POINT, featureNum));
        };
        for (size_t featureNum : {3, 4, 5}) {
            result.push_back(makeFeature(
                "007", BASE_FEATURE_POS,
                BASE_TIME_POINT + std::chrono::minutes{2}, featureNum));
        };

        // other driver
        for (size_t featureNum : {1, 2, 3, 4, 5}) {
            result.push_back(makeFeature("Stepanov", BASE_FEATURE_POS,
                                         BASE_TIME_POINT, featureNum));
        };

        // second passage
        for (size_t featureNum : {1, 2, 3, 4, 5}) {
            result.push_back(makeFeature(
                "007", BASE_FEATURE_POS,
                BASE_TIME_POINT + std::chrono::hours{24}, featureNum));
        };

        auto txn = testFixture_->pool().masterWriteableTransaction();
        db::FeatureGateway{*txn}.insert(result);
        txn->commit();

        return result;
    }

    std::unique_ptr<TestFixture> testFixture_;
};

    TEST_F(UnpublishTest, testPassage)
    {
        auto features = fillData();
        auto txn = testFixture_->pool().masterReadOnlyTransaction();
        EXPECT_EQ(
            tool::load(*txn, Mode::Ride, features.front().id(), 0).size(),
            6u);
        EXPECT_EQ(
            tool::load(*txn, Mode::Ride, features.back().id(), 0).size(), 5u);
        EXPECT_EQ(
            tool::load(*txn, Mode::User, features.front().id(), 0).size(),
            11u);
    }

} // tests
} // tool
} // mrc
} // maps
