#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/mds_path.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/coord_recognition/include/coord_recognition.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/graph_matcher_adapter/include/compact_graph_matcher_adapter.h>
#include <util/system/env.h>

#include <opencv2/opencv.hpp>

extern "C" {
#include <libavformat/avformat.h>
#include <libavcodec/avcodec.h>
#include <libavutil/avutil.h>
#include <libavutil/pixdesc.h>
#include <libswscale/swscale.h>
}

#include <iostream>
#include <mutex>
#include <list>

namespace {

class FFMpegVideoFileReader {
public:
    FFMpegVideoFileReader()
    {
        bufferFrame = av_frame_alloc();

    }
    FFMpegVideoFileReader(const std::string& filePath)
    {
        bufferFrame = av_frame_alloc();
        open(filePath);
    }
    ~FFMpegVideoFileReader() {
        close();
        av_frame_free(&bufferFrame);
    }

    bool open(const std::string& filePath) {
        initializeFFMpeg();

        close();

        int ret = avformat_open_input(&avFormatContext, filePath.c_str(), nullptr, nullptr);
        if (0 > ret) {
            WARN() << "Problem in avforamt_open_input(\"" << filePath << "\"): error =" << ret;
            return false;
        }

        if (!initVideoStream()) {
            avformat_close_input(&avFormatContext);
            return false;
        }

        swsContext = sws_getCachedContext(
            swsContext, videoStream->codec->width, videoStream->codec->height, videoStream->codec->pix_fmt,
            videoStream->codec->width, videoStream->codec->height, AV_PIX_FMT_BGR24, SWS_BICUBIC, nullptr, nullptr, nullptr);
        if (!swsContext) {
            WARN() << "Problem in sws_getCachedContext";
            avcodec_close(videoStream->codec);
            avformat_close_input(&avFormatContext);
            return false;
        }

        frameIndex = 0;
        isEOF = false;
        return true;
    }
    bool restart() {
        REQUIRE(isOpened(),
            "Unable to restart not opened file");
        std::string filePath = avFormatContext->filename;
        close();
        return open(filePath);
    }
    void close() {
        frameIndex = -1;
        fps = 0.0;
        isEOF = true;

        frames.clear();
        if (nullptr != videoStream)
            avcodec_close(videoStream->codec);
        if (nullptr != avFormatContext)
            avformat_close_input(&avFormatContext);
    }
    bool isOpened() const {
        return (nullptr != avFormatContext);
    }

    double getFPS() const {
        return fps;
    }

    int getFrameIndex() const {
        return frameIndex;
    }

    cv::Mat read() {
        REQUIRE(isOpened(),
            "Unable to read frame from not opened file");
        if (frames.empty()) {
            grabFrames();
        }
        if (frames.empty())
            return cv::Mat();
        cv::Mat result = frames.front();
        frames.pop_front();
        frameIndex++;
        return result;
    }
private:
    AVFormatContext* avFormatContext = nullptr;
    AVStream* videoStream = nullptr;
    double fps = 0.;
    SwsContext* swsContext = nullptr;
    AVFrame* bufferFrame = nullptr;
    std::list<cv::Mat> frames;
    int frameIndex = -1;
    bool isEOF = true;
private:
    bool initVideoStream() {
        REQUIRE(avFormatContext != nullptr,
            "Unable to init video stream before avFormatContext initialized");

        int ret = avformat_find_stream_info(avFormatContext, nullptr);
        if (ret < 0) {
            WARN() << "Problem in avformat_find_stream_info:  error =" << ret;
            return false;
        }

        AVCodec* videoCodec = nullptr;
        int videoStreamIdx = av_find_best_stream(avFormatContext, AVMEDIA_TYPE_VIDEO, -1, -1, &videoCodec, 0);
        if (0 > videoStreamIdx) {
            WARN() << "Problem in av_find_best_stream:  error =" << videoStreamIdx;
            return false;
        }

        videoStream = avFormatContext->streams[videoStreamIdx];
        ret = avcodec_open2(videoStream->codec, videoCodec, nullptr);
        if (0 > ret) {
            WARN() << "Problem in avcodec_open2: error =" << ret;
            return false;
        }

        fps = av_q2d(videoStream->codec->framerate);
        return true;
    }

    bool readPacket(AVPacket &packet) {
        REQUIRE(isOpened(),
            "Unable to read packet from not opened file");
        for (;!isEOF;) {
            int ret = av_read_frame(avFormatContext, &packet);
            if (AVERROR_EOF == ret) {
                isEOF = true;
                break;
            }
            if (ret < 0) {
                WARN() << "Problem in av_read_frame: error =" << ret;
                return false;
            }
            if (ret == 0 && packet.stream_index == videoStream->index)
            {
                break;
            }
        }

        if (isEOF) {
            av_init_packet(&packet);
            packet.data = nullptr;
            packet.size = 0;
        }
        return true;
    }

    cv::Mat bufferFrameAsMat() {
        cv::Mat frame(videoStream->codec->height, videoStream->codec->width, CV_8UC3);

        uint8_t *const dst[AV_NUM_DATA_POINTERS ] = {(uint8_t *const)frame.datastart};
        const int dstStride[AV_NUM_DATA_POINTERS ] = {(int)frame.step[0]};

        sws_scale(swsContext, bufferFrame->data, bufferFrame->linesize, 0, bufferFrame->height, dst, dstStride);
        return frame;
    }

    void grabFrames() {
        if (isEOF)
            return;

        for (;frames.empty();) {
            AVPacket packet;

            if (!readPacket(packet))
                break;
            AVPacket origPacket = packet;
            for (;;)
            {
                int gotPicture = 0;
                int ret = avcodec_decode_video2(videoStream->codec, bufferFrame, &gotPicture, &packet);
                if (ret < 0) {
                    WARN() << "Error decoding video frame " << av_err2str(ret);
                    break;
                }
                if (0 != gotPicture) {
                    frames.emplace_back(bufferFrameAsMat());
                }

                packet.data += ret;
                packet.size -= ret;

                if (packet.size <= 0)
                    break;
            }
            av_free_packet(&origPacket);
        }
    }
private:
    static void initializeFFMpeg() {
        static bool initialized = false;
        static std::mutex mtx;
        std::lock_guard lock{mtx};
        if (!initialized) {
            initialized = true;
            av_register_all();
            av_log_set_level(AV_LOG_QUIET);
        }
    }
};

constexpr double THRESHOLD_CONFIDENCE = 0.55;
constexpr int MAX_INVALID_FRAMES_FOR_SPEED_APPROXIMATE_COUNT = 50;
constexpr double MIN_SHIFT_METERS_FOR_SPEED_APPROXIMATE = 10.;

struct FrameCoordinate {
    int frameIndex;
    maps::chrono::TimePoint timePoint;
    maps::geolib3::Point2 coord;
    double confidence;
};

std::vector<FrameCoordinate> recognizeCoordintesOnVideo(
    FFMpegVideoFileReader& videoReader,
    const maps::chrono::TimePoint& startTimePoint)
{
    std::vector<FrameCoordinate> result;

    maps::mrc::coord_recognition::CoordRecognition recog;
    const double videoFPS = videoReader.getFPS();
    cv::Mat frame;
    size_t goodFrames = 0;
    for (;;) {
        frame = videoReader.read();
        if (frame.empty())
            break;
        maps::mrc::coord_recognition::Coordinates
            recognized = recog.recognize(frame);
        if (recognized.second < DBL_EPSILON || recognized.first < DBL_EPSILON)
            recognized.confidence = 0.0;
        FrameCoordinate frameCoordinate = {
            videoReader.getFrameIndex(),
            startTimePoint + std::chrono::duration<int, std::milli>(int(videoReader.getFrameIndex() * 1000. / videoFPS)),
            {recognized.second, recognized.first},
            recognized.confidence
        };
        if (0 == (videoReader.getFrameIndex() % 5000))
            INFO() << "Processed " << videoReader.getFrameIndex() << " frames";
        if (recognized.confidence > THRESHOLD_CONFIDENCE)
            goodFrames++;
        result.emplace_back(frameCoordinate);
    }
    INFO() << "Good frames: " << goodFrames << " from " << result.size();
    return result;
}

struct FramePart {
    size_t framesCnt;
    size_t lastValid;
    int label;
};

void removeBigOutliers(
    std::vector<FrameCoordinate>& coordinates,
    double fps)
{
    // so big, because gps coord change roughly
    constexpr double THRESHOLD_METER_IN_SECOND = 800.;
    if (0 == coordinates.size())
        return;

    std::vector<int> labels(coordinates.size(), -1);
    std::vector<FramePart> frameParts;
    int labelsCnt = 0;
    for (size_t i = 0; i < coordinates.size(); i++) {
        if (coordinates[i].confidence < THRESHOLD_CONFIDENCE)
            continue;
        int framePartIdx = (int)frameParts.size() - 1;
        for (; 0 <= framePartIdx; framePartIdx--) {
            const int lastValid = frameParts[framePartIdx].lastValid;
            const double dist = maps::geolib3::geoDistance(coordinates[lastValid].coord, coordinates[i].coord);
            if ((i - lastValid) * THRESHOLD_METER_IN_SECOND / fps > dist) {
                break;
            }
        }
        if (-1 != framePartIdx) {
            frameParts[framePartIdx].framesCnt ++;
            frameParts[framePartIdx].lastValid = i;
            labels[i] = frameParts[framePartIdx].label;
            if (framePartIdx != (int)frameParts.size() - 1) {
                frameParts.push_back(frameParts[framePartIdx]);
                frameParts.erase(frameParts.begin() + framePartIdx);
            }
        } else {
            FramePart part;
            part.framesCnt = 1;
            part.lastValid = i;
            part.label = labelsCnt;
            frameParts.emplace_back(part);
            labels[i] = labelsCnt;
            labelsCnt++;
        }
    }

    int mainPart = -1;
    for (size_t i = 0; i < frameParts.size(); i++) {
        INFO() << i << "."
                << "framesCnt: " << frameParts[i].framesCnt
                << " label: " << frameParts[i].label;
        if (-1 == mainPart || frameParts[mainPart].framesCnt < frameParts[i].framesCnt)
            mainPart = i;
    }

    int mainPartLabel = -1;
    if (0 <= mainPart)
        mainPartLabel = frameParts[mainPart].label;

    for (size_t i = 0; i < coordinates.size(); i++) {
        if (coordinates[i].confidence < THRESHOLD_CONFIDENCE ||
            labels[i] != mainPartLabel) {
            coordinates[i].confidence = 0.;
            continue;
        }
    }
}

void interpolateBetween(
    std::vector<FrameCoordinate>& coordinates,
    int fromIndex, int toIndex)
{
    if (fromIndex >= toIndex)
        return;
    double x = coordinates[fromIndex].coord.x();
    double y = coordinates[fromIndex].coord.y();
    const double xTo = coordinates[toIndex].coord.x();
    const double yTo = coordinates[toIndex].coord.y();
    const double dx = (xTo - x) / (double)(toIndex - fromIndex);
    const double dy = (yTo - y) / (double)(toIndex - fromIndex);
    const double confidence =
        coordinates[fromIndex].confidence * coordinates[toIndex].confidence;
    for (int j = fromIndex + 1; j < toIndex; j++) {
        x += dx;
        y += dy;
        coordinates[j].coord = maps::geolib3::Point2(x, y);
        coordinates[j].confidence = confidence;
    }
}

void interpolateFirstFrames(
    std::vector<FrameCoordinate>& coordinates,
    int firstValidFrameIdx)
{

    if (0 >= firstValidFrameIdx)
        return;
    int i = firstValidFrameIdx + 1;
    int invalidFrameSuccessively = 0;
    for (;i < (int)coordinates.size() &&
          invalidFrameSuccessively < MAX_INVALID_FRAMES_FOR_SPEED_APPROXIMATE_COUNT; i++)
    {
        if (coordinates[i].confidence < THRESHOLD_CONFIDENCE) {
             invalidFrameSuccessively++;
             continue;
        }
        invalidFrameSuccessively = 0;
        if (MIN_SHIFT_METERS_FOR_SPEED_APPROXIMATE <
                maps::geolib3::geoDistance(coordinates[firstValidFrameIdx].coord, coordinates[i].coord)) {
            break;
        }
    }
    if (invalidFrameSuccessively == MAX_INVALID_FRAMES_FOR_SPEED_APPROXIMATE_COUNT)
        return;
    const double coef = (double)firstValidFrameIdx / (double)(i - firstValidFrameIdx);
    const maps::geolib3::Vector2 shift = coordinates[firstValidFrameIdx].coord - coordinates[i].coord;
    coordinates[0].coord = coordinates[firstValidFrameIdx].coord + coef * shift;
    coordinates[0].confidence = coordinates[i].confidence;
    interpolateBetween(coordinates, 0, firstValidFrameIdx);
    coordinates[0].confidence = coordinates[firstValidFrameIdx].confidence * coordinates[i].confidence;
}


void interpolateLastFrames(
    std::vector<FrameCoordinate>& coordinates,
    int lastValidFrameIdx)
{
    if (lastValidFrameIdx >= (int)coordinates.size() - 1)
        return;
    int i = lastValidFrameIdx - 1;
    int invalidFrameSuccessively = 0;
    for (; 0 <= i &&
           invalidFrameSuccessively < MAX_INVALID_FRAMES_FOR_SPEED_APPROXIMATE_COUNT; i--)
    {
        if (coordinates[i].confidence < THRESHOLD_CONFIDENCE) {
             invalidFrameSuccessively++;
             continue;
        }
        invalidFrameSuccessively = 0;
        if (MIN_SHIFT_METERS_FOR_SPEED_APPROXIMATE <
                maps::geolib3::geoDistance(coordinates[lastValidFrameIdx].coord, coordinates[i].coord)) {
            break;
        }
    }
    if (invalidFrameSuccessively == MAX_INVALID_FRAMES_FOR_SPEED_APPROXIMATE_COUNT)
        return;
    const double coef = ((double)coordinates.size() - lastValidFrameIdx) / (double)(lastValidFrameIdx - i);
    const maps::geolib3::Vector2 shift = coordinates[lastValidFrameIdx].coord - coordinates[i].coord;
    coordinates.back().coord = coordinates[lastValidFrameIdx].coord + coef * shift;
    coordinates.back().confidence = coordinates[i].confidence;
    interpolateBetween(coordinates, lastValidFrameIdx, coordinates.size() - 1);
    coordinates.back().confidence = coordinates[lastValidFrameIdx].confidence * coordinates[i].confidence;
}

void interpolate(
    std::vector<FrameCoordinate>& coordinates)
{
    constexpr int MAX_INVALID_FRAMES_COUNT = 100;

    int firstValidFrameIdx = -1;
    int lastValidFrameIdx = -1;
    for (size_t i = 0; i < coordinates.size(); i++) {
        if (coordinates[i].confidence > THRESHOLD_CONFIDENCE) {
            if (-1 == lastValidFrameIdx) {
                lastValidFrameIdx = i;
                firstValidFrameIdx = i;
            } else {
                const size_t invalidFramesCnt = i - lastValidFrameIdx - 1;
                if (1 <= invalidFramesCnt && invalidFramesCnt < MAX_INVALID_FRAMES_COUNT) {
                    interpolateBetween(coordinates, lastValidFrameIdx, i);
                }
                lastValidFrameIdx = i;
            }
        }
    }
    interpolateFirstFrames(coordinates, firstValidFrameIdx);
    interpolateLastFrames(coordinates, lastValidFrameIdx);
}

std::vector<FrameCoordinate> thinOut(
    const std::vector<FrameCoordinate>& coordinates,
    double fps)
{
    constexpr double MIN_TIMEOUT_SECONDS = 0.75;
    constexpr double MIN_DISTANCE_METERS = 10;

    constexpr double MAX_WITHOUT_FRAME_TIMEOUT_SECONDS = 2.;

    const double minFramesCount = fps * MIN_TIMEOUT_SECONDS;
    const double maxFramesCount = fps * MAX_WITHOUT_FRAME_TIMEOUT_SECONDS;

    int lastSavedFrameIdx = -1;
    int lastValidFrameIdx = -1;
    std::vector<FrameCoordinate> result;
    for (size_t i = 0; i < coordinates.size(); i++) {
        if (coordinates[i].confidence > THRESHOLD_CONFIDENCE) {
            if (-1 == lastSavedFrameIdx) {
                lastSavedFrameIdx = i;
                result.push_back(coordinates[i]);
            } else {
                if (minFramesCount < i - lastSavedFrameIdx)
                {
                    const double dist = maps::geolib3::geoDistance(
                        coordinates[lastSavedFrameIdx].coord,
                        coordinates[i].coord);
                    if (MIN_DISTANCE_METERS < dist) {
                        lastSavedFrameIdx = i;
                        result.push_back(coordinates[i]);
                    }
                }
            }
            lastValidFrameIdx = i;
        } else {
            if (maxFramesCount < i - lastSavedFrameIdx) {
                if (lastSavedFrameIdx >= lastValidFrameIdx) {
                    lastSavedFrameIdx = -1;
                    lastValidFrameIdx = -1;
                } else {
                    lastSavedFrameIdx = lastValidFrameIdx;
                    result.push_back(coordinates[lastValidFrameIdx]);
                }
            }
        }
    }
    return result;
}

/*
    coordinates sorted by frameIndex (=~ timePoint)
*/
void matchToGraph(
    const std::string& graphFolderPath,
    std::vector<FrameCoordinate>& coordinates)
{
    maps::mrc::adapters::CompactGraphMatcherAdapter graphMatcherAdapter_(graphFolderPath);
    maps::mrc::db::TrackPoints trackPoints;
    maps::mrc::db::TrackPoint tp;
    for (size_t i = 0; i < coordinates.size(); i++)
    {
        const FrameCoordinate& coord = coordinates[i];
        tp.setTimestamp(coord.timePoint);
        tp.setGeodeticPos(coord.coord);
        trackPoints.push_back(tp);
    }

    maps::mrc::adapters::TrackSegments segments = graphMatcherAdapter_.match(trackPoints);
    size_t coordIdx = 0;
    size_t segmIdx = 0;
    for (;;)
    {
        FrameCoordinate& coord = coordinates[coordIdx];
        const maps::mrc::adapters::TrackSegment& segment = segments[segmIdx];
        if (coord.timePoint < segment.startTime) {
            coordIdx++;
            if (coordIdx >= coordinates.size())
                break;
        } else if (coord.timePoint < segment.endTime) {
            const double alpha =
                static_cast<double>((coord.timePoint - segment.startTime).count())
                /
                static_cast<double>((segment.endTime - segment.startTime).count());
            coord.coord = segment.segment.pointByPosition(alpha);
            coordIdx++;
            if (coordIdx >= coordinates.size())
                break;
        } else {
            segmIdx++;
            if (segmIdx >= segments.size())
                break;
        }
    }
}

maps::mrc::db::Features
convertToFeatures(
    const std::string& sourceID,
    maps::mrc::db::Dataset dataset,
    const std::vector<FrameCoordinate>& videoCoordinates)
{
    maps::mrc::db::Features features;
    maps::geolib3::Heading heading(0.);
    for (size_t i = 0; i < videoCoordinates.size(); i++) {
        const FrameCoordinate& frameCoordinate = videoCoordinates[i];
        if (i < videoCoordinates.size() - 1)
        {
            heading = maps::geolib3::Direction2(
                        maps::geolib3::convertGeodeticToMercator(
                            maps::geolib3::Segment2(frameCoordinate.coord, videoCoordinates[i + 1].coord)))
                                .heading();
        }
        features
            .emplace_back(
                sourceID,
                frameCoordinate.coord,
                heading,
                maps::chrono::formatSqlDateTime(frameCoordinate.timePoint),
                maps::mds::Key("", ""),
                dataset)
            .setAutomaticShouldBePublished(true);
    }
    return features;
}

void uploadFeatures(
    maps::pgpool3::Pool& pool,
    maps::mds::Mds& mds,
    FFMpegVideoFileReader& videoReader,
    const std::string& sourceID,
    maps::mrc::db::Dataset dataset,
    const std::vector<FrameCoordinate>& videoCoordinates)
{
    maps::mrc::db::Features features =
        convertToFeatures(sourceID, dataset, videoCoordinates);

    REQUIRE(videoCoordinates.size() == features.size(),
        "Amount of features must be same with coordinates");

    auto txn = pool.masterWriteableTransaction();
    maps::mrc::db::FeatureGateway featureGateway(*txn);
    featureGateway.insert(features);

    REQUIRE(videoReader.restart(),
        "Unable to restart video file for add images to database");

    size_t featureIdx = 0;
    cv::Mat frame;
    for (;;) {
        frame = videoReader.read();
        REQUIRE(!frame.empty(),
            "Unable to read frame: " << videoReader.getFrameIndex() << " from video file");
        if (videoCoordinates[featureIdx].frameIndex != videoReader.getFrameIndex())
            continue;

        std::vector<uchar> buf;
        cv::imencode(".jpg", frame, buf);
        std::string encoded((const char*)(buf.data()), buf.size());

        std::string mdsPath = maps::mrc::common::makeMdsPath(
            maps::mrc::common::MdsObjectSource::Imported,
            maps::mrc::common::MdsObjectType::Image,
            features[featureIdx].id());

        maps::mds::PostResponse resp = mds.post(mdsPath, encoded);
        features[featureIdx]
            .setMdsKey(resp.key())
            .setSize(frame.cols, frame.rows)
            .setQuality(1.0)
            .setPrivacy(maps::mrc::db::FeaturePrivacy::Restricted)
            .setAutomaticShouldBePublished(true);

        featureIdx ++;
        if (featureIdx >= features.size())
            break;
    }
    featureGateway.update(features, maps::mrc::db::UpdateFeatureTxn::Yes);
    txn->commit();
}

}

int main(int argc, const char** argv) try {
    maps::cmdline::Parser parser("Launch frames from video publisher");

    maps::cmdline::Option<std::string> inputPath = parser.string("input")
        .required()
        .help("Path to input video file");

    maps::cmdline::Option<std::string> startTime = parser.string("start-time")
        .defaultValue("2018-07-01 12:00:00.000000+03")
        .help("Timestep for first frame");

    maps::cmdline::Option<std::string> graphFolderPath = parser.string("graph-folder")
        .defaultValue("")
        .help("Path to folder with data for graph adapter");

    maps::cmdline::Option<std::string> sourceIDPrefix = parser.string("source-id")
        .defaultValue("VIDEO_FILE_")
        .help("Source id with this prefix will be set to the feature (default: VIDEO_FILE_)");

    maps::cmdline::Option<std::string> mrcConfigPath = parser.string("mrc-config")
        .help("Path to mrc config");

    maps::cmdline::Option<std::string> secretVersion = parser.string("secret-version")
        .help("version for secrets from yav.yandex-team.ru");

    maps::cmdline::Option<bool> dryRunParam = parser.flag("dry-run")
        .help("do not save changes to database");

    maps::cmdline::Option<std::string> outputCoordPath = parser.string("outcoord")
        .defaultValue("")
        .help("Path to file for coordinate save");

    parser.parse(argc, const_cast<char**>(argv));

    if (GetEnv("TF_CPP_MIN_VLOG_LEVEL").empty() &&
        GetEnv("TF_CPP_MIN_LOG_LEVEL").empty())
        SetEnv("TF_CPP_MIN_LOG_LEVEL", "99"); // Silence!

    const maps::mrc::common::Config mrcConfig =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);

    FFMpegVideoFileReader videoReader(inputPath);
    REQUIRE(videoReader.isOpened(),
        "Unable to open file: \"" << inputPath << "\"");

    const double videoFPS = videoReader.getFPS();

    maps::chrono::TimePoint startTimePoint = maps::chrono::parseSqlDateTime(startTime);

    INFO() << "Recognize coordinates on the video (fps: " << videoFPS << ")";
    std::vector<FrameCoordinate> videoCoordinates = recognizeCoordintesOnVideo(videoReader, startTimePoint);

    if (outputCoordPath != "") {
        std::ofstream ofs(outputCoordPath);
        for (size_t i = 0; i < videoCoordinates.size(); i++) {
            ofs << i << " "
                << videoCoordinates[i].coord.x() << " "
                << videoCoordinates[i].coord.y() << " "
                << videoCoordinates[i].confidence << std::endl;
        }
    }
    removeBigOutliers(videoCoordinates, videoFPS);
    interpolate(videoCoordinates);
    videoCoordinates = thinOut(videoCoordinates, videoFPS);
    INFO() << "After thinOut: " << videoCoordinates.size();
    if (0 == videoCoordinates.size()) {
        WARN() << "There are no valid coordinates found.";
        return EXIT_SUCCESS;
    }
    INFO() << "Match points to the road graph";
    matchToGraph(graphFolderPath, videoCoordinates);
    INFO() << "There are " << videoCoordinates.size()
           << " frames to upload to database";
    std::string sourceID = sourceIDPrefix +
        std::to_string(std::hash<std::string>{}(inputPath));
    INFO() << "Upload features with source id: " << sourceID;

    if (dryRunParam)
        return EXIT_SUCCESS;

    maps::wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());
    maps::mds::Mds mds = mrcConfig.makeMdsClient();
    uploadFeatures(mrc.pool(), mds, videoReader, sourceID,
        maps::mrc::db::Dataset::Video, videoCoordinates);
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
