#include "tool.h"

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/vault_boy/include/secrets.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/libs/common/include/exception.h>

namespace {

/**
 * @see
 * https://yav.yandex-team.ru/secret/sec-01cp2k8gbhycereqjrqg2va311/explore/versions
 */
const auto CONFIG_SECRET = "sec-01cp2k8gbhycereqjrqg2va311";

/**
 * @see
 * https://a.yandex-team.ru/arc/trunk/arcadia/maps/wikimap/mapspro/services/mrc/libs/config/cfg/t-config.production.xml
 */
const auto CONFIG_PATH = "../../libs/config/cfg/t-config.production.xml";

}  // anonymous namespace

int main(int argc, char* argv[])
try {
    maps::cmdline::Parser parser;
    auto configPath = parser.string("config")
            .help("path to mrc config")
            .defaultValue(CONFIG_PATH);
    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru")
            .defaultValue(CONFIG_SECRET);
    auto ytTable = parser.string("yt_table")
                       .required()
                       .help(
                           "table in "
                           "https://yt.yandex-team.ru/hahn/navigation"
                           "?path=//home/maps/core/garden/stable/ymapsdf");
    auto adId = parser.size_t("ad_id").required().help(
        "ID of the unit of administrative division");
    auto startTime =
        parser.string("start-time").help("lower limit for photos creation time");
    auto endTime =
        parser.string("end-time").help("upper limit for photos creation time");
    auto sourceId =
        parser.string("source-id").help("photos source id filter");
    auto outDir = parser.string("out_dir").required().help("output directory");
    parser.parse(argc, argv);
    auto cfg = maps::mrc::common::Config(
        maps::vault_boy::loadContextWithYaVault(secretVersion), configPath);

    auto filter =
        maps::mrc::signs_dumper::FilterOptions{
            .geodeticBoundary = maps::mrc::signs_dumper::loadAdGeom(ytTable, adId)
        };

    if (startTime.defined()) {
        filter.startTime = maps::chrono::parseIsoDateTime(startTime);
    }

    if (endTime.defined()) {
        filter.endTime = maps::chrono::parseIsoDateTime(endTime);
    }

    if (sourceId.defined()) {
        filter.sourceId = sourceId;
    }

    std::string resultDir = outDir + "/" + ytTable + "_" + std::to_string(adId);
    maps::mrc::signs_dumper::run(cfg, filter, resultDir);
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
