#include "tool.h"

#include <contrib/libs/geos/include/geos/geom/MultiPolygon.h>
#include <contrib/libs/geos/include/geos/geom/Polygon.h>
#include <contrib/libs/geos/include/geos/io/WKBReader.h>
#include <mapreduce/yt/interface/client.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/parallel_for_each.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/sign.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/sign_gateway.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/common/include/make_batches.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/prepared_polygon.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <boost/filesystem.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <util/generic/iterator_range.h>

#include <fstream>
#include <sstream>
#include <string_view>
#include <variant>

namespace maps::mrc::signs_dumper {
namespace {

constexpr int BATCH_SIZE = 100;

geolib3::MultiPolygon2 hexWKBToMultiPolygon(std::string_view hexWKB)
{
    std::stringstream ss(std::string{hexWKB});
    std::unique_ptr<geos::geom::Geometry> geom(
        geos::io::WKBReader().readHEX(ss));
    if (auto ptr = dynamic_cast<geos::geom::Polygon*>(geom.get())) {
        auto polygon = geolib3::internal::geos2geolibGeometry(ptr);
        return geolib3::MultiPolygon2({std::move(polygon)});
    }
    if (auto ptr = dynamic_cast<geos::geom::MultiPolygon*>(geom.get())) {
        return geolib3::internal::geos2geolibGeometry(ptr);
    }
    throw RuntimeError{} << "Incorrect geometry type";
}

bool isValidType(traffic_signs::TrafficSign)
{
    return true;
    /* todo:
    switch (type) {
        case traffic_signs::TrafficSign::ProhibitoryNoParking:
        case traffic_signs::TrafficSign::ProhibitoryNoParkingOrStopping:
        case traffic_signs::TrafficSign::InformationPaidServices:
            return true;
        default:
            return false;
    }
    */
}

using Signs = std::map<db::TId, db::Sign>;

Signs loadSigns(sql_chemistry::Transaction& txn, const geolib3::MultiPolygon2& mercatorGeom)
{
    auto signs = db::SignGateway{txn}.load(
        db::table::Sign::position.intersects(mercatorGeom.boundingBox()));
    INFO() << "loaded " << signs.size() << " signs";
    Signs result;
    auto preparedGeom = geolib3::PreparedPolygon2(mercatorGeom);
    for (const auto& item : signs) {
        if (isValidType(item.type()) &&
            spatialRelation(preparedGeom,
                            item.mercatorPos(),
                            geolib3::SpatialRelation::Intersects)) {
            result.insert({item.id(), item});
        }
    }
    INFO() << "filtered " << result.size() << " signs";
    return result;
}

using SignFeatures = std::unordered_multimap<db::TId, db::SignFeature>;

SignFeatures loadSignFeatures(pgpool3::Pool& pool, const Signs& signs)
{
    INFO() << "handling signs";
    SignFeatures result;
    size_t counter = 0;
    auto batches = maps::common::makeBatches(signs, BATCH_SIZE);
    common::parallelForEach(
        batches.begin(),
        batches.end(),
        [&](std::mutex& guard, const auto& batch) {
            auto ids = batch | boost::adaptors::map_keys;
            auto signFeatures =
                db::SignFeatureGateway{*pool.slaveTransaction()}.load(
                    db::table::SignFeature::signId.in(
                        {ids.begin(), ids.end()}));

            std::lock_guard lock{guard};
            for (const auto& item : signFeatures) {
                result.insert({item.signId(), item});
            }
            counter += std::distance(ids.begin(), ids.end());
            INFO() << "handled signs: " << counter;
        });
    INFO() << "loaded " << result.size() << " feature signs";
    return result;
}

using Features = std::unordered_map<db::TId, db::Feature>;

Features loadFeatures(pgpool3::Pool& pool,
                      const SignFeatures& signFeatures,
                      const FilterOptions& filter)
{
    auto ids =
        signFeatures | boost::adaptors::map_values |
        boost::adaptors::transformed([&](const db::SignFeature& signFeature) {
            return signFeature.featureId();
        });
    std::set<db::TId> idSet{ids.begin(), ids.end()};
    INFO() << "database features to load: " << idSet.size();

    Features result;
    auto batches = maps::common::makeBatches(idSet, BATCH_SIZE);
    common::parallelForEach(
        batches.begin(),
        batches.end(),
        [&](std::mutex& guard, const auto& batch) {
            auto features = db::FeatureGateway{*pool.slaveTransaction()}.load(
                db::table::Feature::id.in({batch.begin(), batch.end()}));

            features.erase(
                std::remove_if(
                    features.begin(),
                    features.end(),
                    [&](const auto& feature) {
                        return
                            !feature.isPublished() ||
                            feature.privacy() == db::FeaturePrivacy::Secret ||
                            (filter.startTime.has_value() &&
                                feature.timestamp() < filter.startTime.value()) ||
                            (filter.endTime.has_value() &&
                                feature.timestamp() > filter.endTime.value()) ||
                            (filter.sourceId.has_value() &&
                                feature.sourceId() != filter.sourceId.value());
                    }),
                features.end());

            std::lock_guard lock{guard};
            for (const auto& item : features) {
                result.insert({item.id(), item});
            }
            INFO() << "loaded database features: " << result.size();
        });
    return result;
}

struct SignManipulator {
    const db::Sign& sign;
    const SignFeatures& signFeatures;
    const Features& features;

    std::string makeFeatureUrl(const db::SignFeature& signFeature, const db::Feature& feature) const
    {
        auto box = revertByImageOrientation(signFeature.imageBox(),
                                            feature.size(),
                                            feature.orientation());
        std::ostringstream os;
        os << "https://core-nmaps-mrc-browser.maps.yandex.ru/"
                "feature/"
            << signFeature.featureId()
            << "/image?boxes=" << box.minX() << "," << box.minY()
            << "," << box.maxX() << "," << box.maxY();
        return os.str();
    }

    void dumpFeature(json::ObjectBuilder b, const db::SignFeature& signFeature, const db::Feature& feature) const
    {
        b["geometry"] << geolib3::geojson(feature.geodeticPos());
        b["heading"] = (int)feature.heading().value();
        b["date"] = chrono::formatIsoDateTime(feature.timestamp());
        b["url"] = makeFeatureUrl(signFeature, feature);
    }

    void json(json::ObjectBuilder b) const
    {
        b["type"] = "Feature";
        b["id"] = sign.id();
        b["geometry"] << geolib3::geojson(sign.geodeticPos());
        b["properties"] = [&](json::ObjectBuilder b) {
            b["type"] = toString(sign.type());
            b["heading"] = (int)sign.heading().value();
            b["photos"] << [&](json::ArrayBuilder b) {
                for (const auto& signFeature : MakeIteratorRange(signFeatures.equal_range(sign.id()))) {
                    const auto& feature = features.at(signFeature.second.featureId());
                    b << [&](json::ObjectBuilder obj) {
                        dumpFeature(obj, signFeature.second, feature);
                    };
                }
            };
        };
    }
};

void toJson(const std::string& outDir,
            const Signs& signs,
            const Features& features,
            const SignFeatures& signFeatures)
{
    std::ofstream os(outDir + "/result.json");
    json::Builder builder(os);
    builder << [&](json::ObjectBuilder b) {
        b["data"] << [&](json::ObjectBuilder b) {
            b["type"] = "FeatureCollection";
            b["features"] << [&](json::ArrayBuilder b) {
                for (const auto& sign : signs) {
                    SignManipulator manip{sign.second, signFeatures, features};
                    b << manip;
                }
            };
        };
    };
}

void removeDanglingRefs(Signs& signs,
            const Features& features,
            SignFeatures& signFeatures)
{
    for (auto signFeatureIt = signFeatures.begin();
            signFeatureIt != signFeatures.end();)
    {
        if (! features.count(signFeatureIt->second.featureId())) {
            signFeatureIt = signFeatures.erase(signFeatureIt);
        } else {
            ++signFeatureIt;
        }
    }

    for (auto signIt = signs.begin(); signIt != signs.end();)
    {
        if (!signFeatures.count(signIt->first)) {
            signIt = signs.erase(signIt);
        } else {
            ++signIt;
        }
    }
}

}  // namespace


geolib3::MultiPolygon2
loadAdGeom(const std::string& ytTable, long long adId)
{
    INFO() << "Connecting to yt: " << ytTable << ", " << adId;
    NYT::IClientPtr client = NYT::CreateClient("hahn");
    NYT::ITransactionPtr txn = client->StartTransaction();
    std::string path =
        "//home/maps/core/garden/stable/ymapsdf/latest/" + ytTable + "/ad_geom";
    NYT::TRichYPath range(path.c_str());
    range.AddRange(NYT::TReadRange().Exact(NYT::TReadLimit().Key(adId)));
    NYT::TTableReaderPtr<NYT::TNode> reader =
        txn->CreateTableReader<NYT::TNode>(range);
    for (; reader->IsValid(); reader->Next()) {
        auto result = hexWKBToMultiPolygon(reader->GetRow()["shape"].AsString());
        auto bbox = result.boundingBox();
        INFO() << "BBOX(" << bbox.minX() << " " << bbox.minY() << ", "
               << bbox.maxX() << " " << bbox.maxY() << ")";
        return result;
    }
    throw RuntimeError("Failed to load AOI");
}

void run(common::Config& cfg,
         const FilterOptions& filterOptions,
         const std::string& outDir)
{
    boost::filesystem::create_directory(outDir);
    auto mercatorGeom = geolib3::convertGeodeticToMercator(filterOptions.geodeticBoundary);
    auto pool = cfg.makePoolHolder();
    auto signs = loadSigns(*pool.pool().slaveTransaction(), mercatorGeom);
    auto signFeatures = loadSignFeatures(pool.pool(), signs);
    auto features = loadFeatures(pool.pool(), signFeatures, filterOptions);
    removeDanglingRefs(signs, features, signFeatures);
    toJson(outDir, signs, features, signFeatures);
}

}  // namespace maps::mrc::signs_dumper
