#include "utils.h"

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <maps/libs/http/include/http.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/contains.h>
#include <maps/libs/geolib/include/distance.h>
#include <yandex/maps/geolib3/sproto.h>

#include <boost/lexical_cast.hpp>

#include <fstream>
#include <memory>

namespace mrc = maps::mrc;
namespace geolib3 = maps::geolib3;
namespace http = maps::http;

using namespace mrc::tasks_analyzer;

namespace {


void checkPolylinesAreAjacent(const geolib3::PolylinesVector& polylines)
{
    for (size_t i = 1; i < polylines.size(); ++i) {
        const auto& prev = polylines[i - 1];
        const auto& cur = polylines[i];
        REQUIRE(prev.pointAt(prev.pointsNumber() - 1) ==
                cur.pointAt(0),
            "targets are not ajacent at" << i);
    }
}


geolib3::Polyline2 join(const geolib3::PolylinesVector& polylines)
{
    geolib3::Polyline2 result;
    for(const auto& polyline : polylines) {
        result.extend(polyline, geolib3::MergeEqualPoints);
    }
    return result;
}


void processTask(const mrc::db::ugc::Task& task,
                const std::string& routerEndpoint,
                int routeRequestPointsBatchSize,
                const std::string& easyviewFile,
                const std::string& geomFilename)
{
    std::string taskName;
    if (!task.names().empty()) {
        taskName = task.names().begin()->second;
    }
    INFO() << "processing task id=" << task.id() << " " << taskName;
    INFO() << "easyviewFile=" << easyviewFile << ", geomFilename=" << geomFilename;

    const mrc::db::ugc::Targets& targets = task.targets();
    auto polylines = makeSortedPolylines(targets);

    std::ofstream targets_out("route.ev");
    targets_out << "!linestyle=blue:4\n";
    savePolylinesToFile(polylines, targets_out, 10);

    checkPolylinesAreAjacent(polylines);
    auto viaPoints = makeRouteViaPointsForTaskTargets(polylines);
    INFO() << "via points: " << viaPoints.size();

    geolib3::PolylinesVector routes;

    http::Client client;

    for (auto pointsBatch : splitToBatches(viaPoints, routeRequestPointsBatchSize, 1))
    {
        try {
        auto routeResponse = makeRouteRequest(client, routerEndpoint, pointsBatch);
        for (const auto& routePart : parseRouteResponse(routeResponse)) {
            routes.push_back(routePart);
        }
        } catch(const maps::Exception& ex) {
            ERROR() << "failed to build route: " << ex;
            std::ofstream out("failed_route.ev");
            out << "!pointstyle=green:blue:4\n";
            size_t idx = 0;
            for (const auto point : pointsBatch) {
                formatInEasyview(out, point, std::to_string(idx));
                idx++;
            }
            throw;
        }
    }

    auto ratio = calcRouteRatio(targets, routes);

    if (ratio < 1.5) {
        return;
    }

    auto plannedRoute = join(polylines);
    auto actualRoute = join(routes);

    if (!easyviewFile.empty()) {
        auto viaPointsInfo = checkViaPointsPassingOrder(plannedRoute, actualRoute, viaPoints);
        saveViaPointsCheck(viaPointsInfo, easyviewFile);
    }

    if (!geomFilename.empty()) {
        geolib3::Polyline2 routePolyline;
        bool first = true;
        for (const auto& polyline : polylines) {
            if (first) {
                routePolyline = polyline;
                first = false;
            } else {
                routePolyline.extend(polyline);
            }
        }

        INFO() << "writing route polyline to " << geomFilename;
        std::ofstream routeProtoFile(geomFilename, std::ios::binary);
        auto routeSproto = geolib3::sproto::encode(routePolyline);
        routeProtoFile << routeSproto;
    }
}


} // namespace

int main(int argc, char** argv)
try {

    maps::cmdline::Parser parser;
    auto configPath = parser.string("config")
        .help("path to configuration");

    auto taskId = parser.num("task");

    auto taskStatus = parser.string("task-status");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto routerEndpoint = parser.string("router")
        .defaultValue("http://core-driving-router.testing.maps.yandex.net/");

    auto routeRequestPointsBatchSize =
        parser.num("points-batch-size")
            .defaultValue(100);

    auto easyviewFilename = parser.string("easyview");

    auto geomFilename = parser.string("geomFilename");

    parser.parse(argc, argv);

    const auto config =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);

    auto poolHolder = config.makePoolHolder();

    auto txn = poolHolder.pool().slaveTransaction();
    maps::sql_chemistry::FiltersCollection filter{maps::sql_chemistry::op::Logical::And};

    if (taskStatus.defined()){
        auto status = boost::lexical_cast<mrc::db::ugc::TaskStatus>(taskStatus);
        filter.add(mrc::db::ugc::table::Task::status == status);
    }
    if (taskId.defined()) {
        auto id = static_cast<mrc::db::TId>(taskId);
        filter.add(mrc::db::ugc::table::Task::id == id);
    }
    mrc::db::ugc::TaskGateway gtw(*txn);
    auto tasks = gtw.load(filter);
    gtw.loadNames(tasks);
    gtw.loadTargets(tasks);

    for (const auto& task : tasks)
    try {

        std::string taskEvFilename = easyviewFilename + std::to_string(task.id());
        processTask(task,
                    routerEndpoint,
                    routeRequestPointsBatchSize,
                    taskEvFilename,
                    geomFilename);


    } catch (const maps::Exception& ex) {
        ERROR() << "Failed to process task " << task.id()
            << ": " << ex;
    }
    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    ERROR() << "Caught unknown exception";
    return EXIT_FAILURE;
}
