#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>

#include <mapreduce/yt/util/ypath_join.h>
#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/interface/client.h>

#include <opencv2/opencv.hpp>

#include <vector>

using namespace maps;
using namespace maps::geolib3;

namespace {

static const TString LAT = "lat";
static const TString LON = "lon";
static const TString X_COORD = "x";
static const TString Y_COORD = "y";
static const TString COND_TYPE = "cond_type";
static const TString COND_SEQ_ID = "cond_seq_id";
static const TString RD_JC_ID = "rd_jc_id";
static const TString ID = "feature_id";
static const TString URL = "url";
static const TString ORIENTATION = "orientation";

static const int64_t TRAFFIC_LIGHT_COND_TYPE = 7;

static const TString LATEST_YMAPSDF_PATH = "//home/maps/core/garden/stable/ymapsdf/latest";

class YMapsDFYTTable {
public:
    static std::vector<NYT::TYPath> getCondYTTablePaths(NYT::ITransactionPtr txn) {
        return getAllRegionsYTTablePaths(txn, "cond");
    }

    static std::vector<NYT::TYPath> getCondRdSeqYTTablePaths(NYT::ITransactionPtr txn) {
        return getAllRegionsYTTablePaths(txn, "cond_rd_seq");
    }

    static std::vector<NYT::TYPath> getRdJcYTTablePaths(NYT::ITransactionPtr txn) {
        return getAllRegionsYTTablePaths(txn, "rd_jc");
    }

private:
    static std::vector<NYT::TYPath> getAllRegionsYTTablePaths(
        NYT::ITransactionPtr txn, const TString& tableName)
    {
        std::vector<NYT::TYPath> tablePaths;
        NYT::TNode::TListType regionList = txn->List(LATEST_YMAPSDF_PATH);
        for (int i = 0; i < regionList.ysize(); i++) {
            TString region = regionList[i].AsString();
            tablePaths.push_back(
                NYT::JoinYPaths(LATEST_YMAPSDF_PATH, region, tableName)
            );
        }
        return tablePaths;
    }
};

class TrafficLightCondExtractorMapper
    : public NYT::IMapper<NYT::TTableReader<NYT::TNode>,
                          NYT::TTableWriter<NYT::TNode>>
{
public:
    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override
    {
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& row = reader->GetRow();
            if (row[COND_TYPE].AsInt64() == TRAFFIC_LIGHT_COND_TYPE) {
                writer->AddRow(row);
            }
        }
    }
};

REGISTER_MAPPER(TrafficLightCondExtractorMapper);

void extractTrafficLightCond(
    NYT::ITransactionPtr txn,
    const NYT::TYPath& outputYTTablePath)
{
    NYT::TMapOperationSpec spec;
    for (const NYT::TYPath& path : YMapsDFYTTable::getCondYTTablePaths(txn)) {
        spec.AddInput<NYT::TNode>(path);
    }
    spec.AddOutput<NYT::TNode>(outputYTTablePath);
    txn->Map(spec, new TrafficLightCondExtractorMapper());
}

class CondRdSeqExtractorReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>>
{
public:
    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override
    {
        constexpr size_t trafficLightTableIndex = 0;
        bool isTrafficLight = false;
        NYT::TNode condSeqNode;
        for (; reader->IsValid(); reader->Next()) {
            size_t tableIndex = reader->GetTableIndex();
            if (trafficLightTableIndex == tableIndex) {
                isTrafficLight = true;
            } else {
                const NYT::TNode& row = reader->GetRow();
                if (!row[RD_JC_ID].IsNull()) {
                    condSeqNode = row;
                }
            }
        }
        if (isTrafficLight) {
            writer->AddRow(condSeqNode);
        }
    }
};

REGISTER_REDUCER(CondRdSeqExtractorReducer);

void extractCondRdSeq(
    NYT::ITransactionPtr txn,
    const NYT::TYPath& inputYTTablePath,
    const NYT::TYPath& outputYTTablePath)
{
    NYT::TTempTable sortedTable(txn);
    txn->Sort(
        NYT::TSortOperationSpec()
            .AddInput(inputYTTablePath)
            .Output(sortedTable.Name())
            .SortBy({COND_SEQ_ID})
    );
    NYT::TReduceOperationSpec spec;
    spec.AddInput<NYT::TNode>(sortedTable.Name());
    for (const NYT::TYPath& path : YMapsDFYTTable::getCondRdSeqYTTablePaths(txn)) {
        spec.AddInput<NYT::TNode>(path);
    }
    spec.AddOutput<NYT::TNode>(outputYTTablePath);
    spec.ReduceBy({COND_SEQ_ID});
    txn->Reduce(spec, new CondRdSeqExtractorReducer());
}

class RdJcCoordExtractorReducer
    : public NYT::IReducer<NYT::TTableReader<NYT::TNode>,
                           NYT::TTableWriter<NYT::TNode>>
{
public:
    void Do(NYT::TTableReader<NYT::TNode>* reader,
            NYT::TTableWriter<NYT::TNode>* writer) override
    {
        constexpr size_t trafficLightTableIndex = 0;
        bool isTrafficLight = false;
        double x;
        double y;
        for (; reader->IsValid(); reader->Next()) {
            size_t tableIndex = reader->GetTableIndex();
            if (trafficLightTableIndex == tableIndex) {
                isTrafficLight = true;
            } else {
                const NYT::TNode& row = reader->GetRow();
                x = row[X_COORD].AsDouble();
                y = row[Y_COORD].AsDouble();
            }
        }
        if (isTrafficLight) {
            writer->AddRow(NYT::TNode()(LON, x)(LAT, y));
        }
    }
};

REGISTER_REDUCER(RdJcCoordExtractorReducer);

void extractTrafficLightPos(
    NYT::ITransactionPtr txn,
    const NYT::TYPath& inputYTTablePath,
    const NYT::TYPath& outputYTTablePath)
{
    NYT::TTempTable sortedTable(txn);
    txn->Sort(
        NYT::TSortOperationSpec()
            .AddInput(inputYTTablePath)
            .Output(sortedTable.Name())
            .SortBy({RD_JC_ID})
    );
    NYT::TReduceOperationSpec spec;
    spec.AddInput<NYT::TNode>(sortedTable.Name());
    for (const NYT::TYPath& path : YMapsDFYTTable::getRdJcYTTablePaths(txn)) {
        spec.AddInput<NYT::TNode>(path);
    }
    spec.AddOutput<NYT::TNode>(outputYTTablePath);
    spec.ReduceBy({RD_JC_ID});
    txn->Reduce(spec, new RdJcCoordExtractorReducer());
}

PointsVector loadTrafficLightsPos(
    NYT::ITransactionPtr txn,
    const NYT::TYPath& ytTablePath)
{
    PointsVector trafficLightsPos;
    NYT::TTableReaderPtr<NYT::TNode> reader
        = txn->CreateTableReader<NYT::TNode>(ytTablePath);
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        Point2 geoPoint(row[LON].AsDouble(), row[LAT].AsDouble());
        Point2 mercPoint = convertGeodeticToMercator(geoPoint);
        trafficLightsPos.push_back(mercPoint);
    }
    return trafficLightsPos;
}

struct TrafficLightCluster {
    PointsVector trafficLightsPos;
    BoundingBox bbox;
};

std::vector<TrafficLightCluster>
clusterizeTrafficLights(const PointsVector& trafficLightsPos) {
    constexpr double CLUSTER_TOLERANCE_METERS = 50.;
    std::vector<BoundingBox> bboxes;
    bboxes.reserve(trafficLightsPos.size());
    for (const Point2& pos : trafficLightsPos) {
        double bboxSizeMerc = toMercatorUnits(CLUSTER_TOLERANCE_METERS, pos);
        bboxes.emplace_back(pos, bboxSizeMerc, bboxSizeMerc);
    }

    std::vector<int> labels;
    int lblsCnt = cv::partition(
        bboxes, labels,
        [](const BoundingBox& lhs, const BoundingBox& rhs) {
            return intersects(lhs, rhs);
        }
    );

    std::vector<TrafficLightCluster> clusters(lblsCnt);
    for (size_t i = 0; i < trafficLightsPos.size(); i++) {
        TrafficLightCluster& cluster = clusters[labels[i]];
        if (cluster.trafficLightsPos.empty()) {
            cluster.bbox = trafficLightsPos[i].boundingBox();
        } else {
            cluster.bbox = expand(cluster.bbox, trafficLightsPos[i]);
        }
        cluster.trafficLightsPos.push_back(trafficLightsPos[i]);
    }

    return clusters;
}

mrc::db::Features extractFeatures(
    pgpool3::Pool& pool,
    const std::vector<TrafficLightCluster>& clusters,
    double searchRadiusMeters)
{
    mrc::db::Features features;
    pgpool3::TransactionHandle txn = pool.slaveTransaction();
    for (const TrafficLightCluster& cluster : clusters) {
        double searchRadiusMerc = toMercatorUnits(searchRadiusMeters, cluster.bbox.center());
        BoundingBox searchBBox = resizeByValue(cluster.bbox, searchRadiusMerc);
        mrc::db::Features clusterFeatures = mrc::db::FeatureGateway(*txn).load(
            mrc::db::table::Feature::isPublished &&
            mrc::db::table::Feature::pos.intersects(searchBBox)
        );

        if (clusterFeatures.empty()) {
            continue;
        }
        StaticGeometrySearcher<Point2, int> searcher;
        for (const Point2& pos : cluster.trafficLightsPos) {
            searcher.insert(&pos, -1);
        }
        searcher.build();
        clusterFeatures.erase(
            std::remove_if(
                clusterFeatures.begin(), clusterFeatures.end(),
                [&](const mrc::db::Feature& feature) {
                    const Point2 mercPos = feature.mercatorPos();
                    auto searchResult = searcher.find(
                        resizeByValue(mercPos.boundingBox(), searchRadiusMerc)
                    );
                    for (auto it = searchResult.first; it != searchResult.second; it++) {
                        if (distance(mercPos, it->geometry()) < searchRadiusMerc) {
                            return false;
                        }
                    }
                    return true;
                }
            ),
            clusterFeatures.end()
        );
        features.insert(
            features.end(),
            clusterFeatures.begin(), clusterFeatures.end()
        );
    }
    return features;
}

void savingFeatures(
    NYT::ITransactionPtr txn,
    mds::Mds& mds,
    const mrc::db::Features& features,
    const NYT::TYPath& outputYTTablePath)
{
    NYT::TTableWriterPtr<NYT::TNode> writer
        = txn->CreateTableWriter<NYT::TNode>(outputYTTablePath);
    for (const mrc::db::Feature& feature : features) {
        writer->AddRow(
            NYT::TNode()
            (ID, feature.id())
            (URL, TString(mds.makeReadUrl(feature.mdsKey())))
            (LAT, feature.geodeticPos().y())
            (LON, feature.geodeticPos().x())
            (ORIENTATION, static_cast<int>(feature.orientation()))
        );
    }
    writer->Finish();
}

} // namespace

int main(int argc, const char** argv)
try {
    NYT::Initialize(argc, argv);

    cmdline::Parser parser("Extracting features around traffic lights");

    cmdline::Option<double> searchRadiusMeters = parser.real("search_radius")
        .required()
        .help("Search radius for traffic lights");

    cmdline::Option<std::string> mrcConfigPath = parser.string("mrc_config")
        .required()
        .help("Path to mrc config");

    cmdline::Option<std::string> secretVersion = parser.string("secret_version")
        .help("version for secrets from yav.yandex-team.ru");

    cmdline::Option<std::string> outputYTTablePath = parser.string("output")
        .required()
        .help("Path to output YT table with features");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Loading mrc config";
    const mrc::common::Config mrcConfig =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);

    INFO() << "Create YT client: yt::hahn";
    NYT::IClientPtr client = NYT::CreateClient("hahn");

    INFO() << "Starting transaction";
    NYT::ITransactionPtr txn = client->StartTransaction();

    INFO() << "Extracting traffic light conds from YMapsDF";
    NYT::TTempTable trafficLightCondTable(txn);
    extractTrafficLightCond(txn, trafficLightCondTable.Name());

    INFO() << "Extracting conds seq";
    NYT::TTempTable condRdSeqTable(txn);
    extractCondRdSeq(txn, trafficLightCondTable.Name(), condRdSeqTable.Name());

    INFO() << "Extracting position of traffic lights";
    NYT::TTempTable trafficLightsPosTable(txn);
    extractTrafficLightPos(
        txn,
        condRdSeqTable.Name(),
        trafficLightsPosTable.Name()
    );

    INFO() << "Loading position of traffic lights from YT";
    PointsVector trafficLightsPos
        = loadTrafficLightsPos(txn, trafficLightsPosTable.Name());
    INFO() << "Loaded " << trafficLightsPos.size() << " traffic lights";

    INFO() << "Clusterizing traffic lights";
    std::vector<TrafficLightCluster> clusters = clusterizeTrafficLights(trafficLightsPos);
    INFO() << "Clusters count: " << clusters.size();

    INFO() << "Search features around clusters in radius " << searchRadiusMeters << " meters";
    wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());
    pgpool3::Pool& pool = mrc.pool();
    mrc::db::Features features = extractFeatures(pool, clusters, searchRadiusMeters);
    INFO() << "Loaded " << features.size() << " features";

    INFO() << "Saving features into YT table: " << outputYTTablePath;
    mds::Mds mds = mrcConfig.makeMdsClient();
    savingFeatures(txn, mds, features, NYT::TYPath(outputYTTablePath));

    INFO() << "Sorting by feature id: " << outputYTTablePath;
    txn->Sort(
        NYT::TSortOperationSpec()
            .AddInput(NYT::TYPath(outputYTTablePath))
            .Output(NYT::TYPath(outputYTTablePath))
            .SortBy({ID})
    );

    txn->Commit();

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
