#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/json/include/value.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>

#include <mapreduce/yt/interface/client.h>

#include <map>
#include <vector>

using namespace maps;
using namespace maps::mrc;

namespace {

constexpr const char* INPUT_VALUES = "inputValues";
constexpr const char* OUTPUT_VALUES = "outputValues";
constexpr const char* FEATURE_ID = "feature_id";
constexpr const char* URL = "url";
constexpr const char* ORIENTATION = "orientation";
constexpr const char* STATE = "state";
constexpr const char* YES_STATE = "yes";

struct TolokaResult {
    size_t yesCount = 0;
    size_t answersCount = 0;
};

std::unordered_map<int64_t, TolokaResult>
loadTolokaResults(const std::string& path) {
    json::Value assignments = json::Value::fromFile(path);

    std::unordered_map<int64_t, TolokaResult> featureIdToResult;
    for (const json::Value& assignment : assignments) {
        int64_t featureId = assignment[INPUT_VALUES][FEATURE_ID].as<int64_t>();
        std::string state = assignment[OUTPUT_VALUES][STATE].as<std::string>();
        featureIdToResult[featureId].answersCount++;
        if (YES_STATE == state) {
            featureIdToResult[featureId].yesCount++;
        }
    }

    return featureIdToResult;
}

} // namespace

int main(int argc, const char** argv)
try {
    NYT::Initialize(argc, argv);

    cmdline::Parser parser("Save toloka results to YT table");

    cmdline::Option<std::string> assignmentsPath = parser.string("assignments_path")
        .required()
        .help("Path to JSON with assignments");

    cmdline::Option<std::string> mrcConfigPath = parser.string("mrc_config")
        .required()
        .help("Path to mrc config");

    cmdline::Option<std::string> secretVersion = parser.string("secret_version")
        .help("version for secrets from yav.yandex-team.ru");

    cmdline::Option<std::string> outputYTTablePath = parser.string("output")
        .required()
        .help("Path to output YT table with results");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Loading mrc config";
    const mrc::common::Config mrcConfig =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);

    INFO() << "Create YT client: yt::hahn";
    NYT::IClientPtr ytClient = NYT::CreateClient("hahn");

    INFO() << "Loading results: " << assignmentsPath;
    std::unordered_map<int64, TolokaResult> featureIdToResult
        = loadTolokaResults(assignmentsPath);
    INFO() << "Loaded results for " << featureIdToResult.size() << " features";

    INFO() << "Filtering accepted features";
    std::vector<int64_t> featureIds;
    for (const auto& [featureId, tolokaResult] : featureIdToResult) {
        if (tolokaResult.yesCount > tolokaResult.answersCount / 2) {
            featureIds.push_back(featureId);
        }
    }
    INFO() << featureIds.size() << " features left";

    INFO() << "Loading feature from MRC db";
    wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());
    pgpool3::Pool& pool = mrc.pool();
    pgpool3::TransactionHandle txn = pool.slaveTransaction();
    mrc::db::Features features
        = mrc::db::FeatureGateway(*txn).loadByIds(std::move(featureIds));

    INFO() << "Saving accepted features into YT table: " << outputYTTablePath;
    mds::Mds mds = mrcConfig.makeMdsClient();
    http::Client httpClient;
    NYT::TTableWriterPtr<NYT::TNode> writer
        = ytClient->CreateTableWriter<NYT::TNode>(NYT::TYPath(outputYTTablePath));
    for (const db::Feature& feature : features) {
        writer->AddRow(
            NYT::TNode()
            (FEATURE_ID, feature.id())
            (URL, TString(mds.makeReadUrl(feature.mdsKey())))
            (ORIENTATION, static_cast<int>(feature.orientation()))
        );
    }
    writer->Finish();

    INFO() << "Sorting table by feature id: " << outputYTTablePath;
    ytClient->Sort(
        NYT::TSortOperationSpec()
            .AddInput(NYT::TYPath(outputYTTablePath))
            .Output(NYT::TYPath(outputYTTablePath))
            .SortBy({FEATURE_ID})
    );

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
