#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/http/include/http.h>

#include <maps/libs/json/include/value.h>

#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_common/include/merge_rects.h>

#include <maps/libs/common/include/base64.h>

#include <mapreduce/yt/interface/client.h>

#include <vector>
#include <unordered_map>

using namespace maps;

namespace {

constexpr const char* RECTANGLE = "rectangle";
constexpr const char* TYPE = "type";
constexpr const char* DATA = "data";
constexpr const char* P1 = "p1";
constexpr const char* P2 = "p2";
constexpr const char* X = "x";
constexpr const char* Y = "y";
constexpr const char* ASSIGNMENT_ID = "assignmentId";
constexpr const char* INPUT_VALUES = "inputValues";
constexpr const char* OUTPUT_VALUES = "outputValues";
constexpr const char* FEATURE_URL = "image";
constexpr const char* RESULT = "result";
constexpr const char* FEATURE_ID = "feature_id";
constexpr const char* IMAGE = "image";
constexpr const char* OBJECTS = "objects";
constexpr const char* MDS_URL = "url";
constexpr const char* ORIENTATION = "orientation";
constexpr const char* BBOX = "bbox";
constexpr const char* STATE = "state";
constexpr const char* IS_NOT_EMPTY = "is_not_empty";
constexpr const char* TRAFFIC_LIGHT = "traffic_light";

cv::Mat loadImage(http::Client& client, const std::string& url) {
    return common::retry(
        [&]() {
            http::Request request(client, http::GET, http::URL(url));
            http::Response response = request.perform();
            if (response.status() == 200) {
                std::vector<uint8_t> bytes = response.readBodyToVector();
                return mrc::common::decodeImage(bytes);
            } else {
                throw maps::RuntimeError()
                    << "Failed to download image: " << url;
            }
        },
        common::RetryPolicy()
            .setTryNumber(6)
            .setInitialCooldown(std::chrono::seconds(1))
            .setCooldownBackoff(2.)
    );
}

// Json format:
// {
//   "type": "rectangle",
//   "data": {
//     "p1": {"x": 0.5, "y": 0.6},
//     "p2": {"x": 0.6, "y": 0.7}
//   }
// }
//
mrc::toloka::common::Rect jsonToRect(const json::Value& value) {
    REQUIRE(
        RECTANGLE == value[TYPE].as<std::string>(),
        "Unknown type: " + value[TYPE].as<std::string>()
    );
    return maps::mrc::toloka::common::Rect(
        geolib3::Point2(value[DATA][P1][X].as<double>(), value[DATA][P1][Y].as<double>()),
        geolib3::Point2(value[DATA][P2][X].as<double>(), value[DATA][P2][Y].as<double>())
    );
}

mrc::toloka::common::Rect resizeRect(
    const cv::Mat& image, const mrc::toloka::common::Rect& normRect)
{
    return mrc::toloka::common::Rect(
        geolib3::Point2(normRect.minX() * image.cols, normRect.minY() * image.rows),
        geolib3::Point2(normRect.maxX() * image.cols, normRect.maxY() * image.rows)
    );
}

std::vector<mrc::toloka::common::Rect> jsonToRects(const json::Value& values) {
    std::vector<mrc::toloka::common::Rect> rects;
    for (const json::Value& value : values) {
        rects.push_back(jsonToRect(value));
    }
    return rects;
}

std::vector<mrc::toloka::common::Rect> resizeRects(
    const cv::Mat& image, const std::vector<mrc::toloka::common::Rect>& normRects)
{
    std::vector<mrc::toloka::common::Rect> rects;
    for (const mrc::toloka::common::Rect& normRect : normRects) {
        rects.push_back(resizeRect(image, normRect));
    }
    return rects;
}

struct Assignment {
    explicit Assignment(const json::Value& value)
        : assignmentId(value[ASSIGNMENT_ID].as<std::string>())
        , featureId(value[INPUT_VALUES][FEATURE_ID].as<uint64_t>())
        , photo(value[INPUT_VALUES][FEATURE_URL].as<std::string>())
    {
        if (value[OUTPUT_VALUES][STATE].as<std::string>() == IS_NOT_EMPTY) {
            normRects = jsonToRects(value[OUTPUT_VALUES][RESULT]);
        }
    }

    std::string assignmentId;
    uint64_t featureId;
    std::string photo;
    std::vector<mrc::toloka::common::Rect> normRects;
};

std::unordered_map<mrc::db::TId, std::vector<Assignment>>
parseAssignments(const std::string& path) {
    json::Value values = json::Value::fromFile(path);
    std::unordered_map<mrc::db::TId, std::vector<Assignment>> featureIdToAssignments;
    for (const json::Value& value : values) {
        Assignment assignment(value);
        featureIdToAssignments[assignment.featureId].push_back(assignment);
    }
    return featureIdToAssignments;
}

NYT::TNode rectsToNode(const std::vector<mrc::toloka::common::Rect>& rects) {
    NYT::TNode objects = NYT::TNode::CreateList();
    for (const mrc::toloka::common::Rect& rect : rects) {
        NYT::TNode object;
        object[TYPE] = TRAFFIC_LIGHT;
        object[BBOX] = NYT::TNode::CreateList()
            .Add(
                NYT::TNode::CreateList()
                    .Add(std::lround(rect.minX()))
                    .Add(std::lround(rect.minY()))
            )
            .Add(
                NYT::TNode::CreateList()
                    .Add(std::lround(rect.maxX()))
                    .Add(std::lround(rect.maxY()))
            );
        objects.Add(object);
    }
    return objects;
}

std::vector<mrc::toloka::common::Rect> aggregateAssignments(
    const cv::Mat& image, const std::vector<Assignment>& assignments)
{
    std::vector<mrc::toloka::common::TolokaAnswer> answers;
    for (const Assignment& assignment : assignments) {
        mrc::toloka::common::TolokaAnswer answer;
        answer.assignment = assignment.assignmentId;
        answer.photo = assignment.photo;
        answer.rects = resizeRects(image, assignment.normRects);
        answers.push_back(answer);
    }
    return aggregatePhoto(answers);
}

} // namespace

int main(int argc, const char** argv)
try {
    NYT::Initialize(argc, argv);

    cmdline::Parser parser("Save toloka results to YT table");

    cmdline::Option<std::string> assignmentsPath = parser.string("assignments_path")
        .required()
        .help("Path to JSON with assignments");

    cmdline::Option<std::string> mrcConfigPath = parser.string("mrc_config")
        .required()
        .help("Path to mrc config");

    cmdline::Option<std::string> secretVersion = parser.string("secret_version")
        .help("version for secrets from yav.yandex-team.ru");

    cmdline::Option<std::string> outputYTTablePath = parser.string("output")
        .required()
        .help("Path to output YT table with results");

    parser.parse(argc, const_cast<char**>(argv));

    INFO() << "Loading assignments: " << assignmentsPath;
    std::unordered_map<mrc::db::TId, std::vector<Assignment>> featureIdToAssignments
        = parseAssignments(assignmentsPath);

    INFO() << "Loading mrc config";
    mrc::common::Config mrcConfig =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, mrcConfigPath);

    INFO() << "Loading features from MRC database";
    std::vector<mrc::db::TId> featureIds;
    for (const auto& [featureId, assignments] : featureIdToAssignments) {
        featureIds.push_back(featureId);
    }
    wiki::common::PoolHolder mrc(mrcConfig.makePoolHolder());
    pgpool3::Pool& pool = mrc.pool();
    pgpool3::TransactionHandle txn = pool.slaveTransaction();
    mrc::db::Features features
        = mrc::db::FeatureGateway(*txn).loadByIds(std::move(featureIds));
    std::sort(
        features.begin(), features.end(),
        [](const mrc::db::Feature& lhs, const mrc::db::Feature& rhs) {
            return lhs.id() < rhs.id();
        }
    );

    INFO() << "Creating MDS client";
    mds::Mds mds = mrcConfig.makeMdsClient();

    INFO() << "Creating YT client: yt::hahn";
    NYT::IClientPtr ytClient = NYT::CreateClient("hahn");

    INFO() << "Saving results into YT table: "<< outputYTTablePath;
    http::Client httpClient;
    NYT::TTableWriterPtr<NYT::TNode> writer = ytClient->CreateTableWriter<NYT::TNode>(
        NYT::TRichYPath(NYT::TYPath(outputYTTablePath))
            .Schema(NYT::TTableSchema()
                .AddColumn(FEATURE_ID, NYT::EValueType::VT_INT64, NYT::ESortOrder::SO_ASCENDING)
                .AddColumn(ORIENTATION, NYT::EValueType::VT_INT64)
                .AddColumn(MDS_URL, NYT::EValueType::VT_STRING)
                .AddColumn(IMAGE, NYT::EValueType::VT_STRING)
                .AddColumn(OBJECTS, NYT::EValueType::VT_ANY)
            )
    );
    for (const mrc::db::Feature& feature : features) {
        std::string mdsURL = mds.makeReadUrl(feature.mdsKey());

        cv::Mat image = loadImage(httpClient, mdsURL);
        image = mrc::common::transformByImageOrientation(image, feature.orientation());
        std::vector<uint8_t> jpegData;
        cv::imencode(".jpg", image, jpegData);

        std::vector<mrc::toloka::common::Rect> rects
            = aggregateAssignments(image, featureIdToAssignments[feature.id()]);

        writer->AddRow(
            NYT::TNode()
                (FEATURE_ID, feature.id())
                (MDS_URL, TString(mdsURL))
                (ORIENTATION, static_cast<int>(feature.orientation()))
                (IMAGE, NYT::TNode(base64Encode(jpegData)))
                (OBJECTS, rectsToNode(rects))
        );
    }

    INFO() << "Finishing";

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    INFO() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    INFO() << e.what();
    return EXIT_FAILURE;
}
catch (...) {
    INFO() << "Caught unknown exception";
    return EXIT_FAILURE;
}
